<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Productos_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->table = 'inv_productos';
        $this->table_cat = 'inv_categorias';
    }

    public function getProductos($incluir_inactivos = false) {
        $this->db->select('p.*, c.nombre as categoria');
        $this->db->from($this->table . ' p');
        $this->db->join($this->table_cat . ' c', 'p.id_categoria = c.id_categoria');
        
        // Solo filtrar por estado activo si no se solicitan los inactivos
        if (!$incluir_inactivos) {
            $this->db->where('p.estado', '1');
        }
        
        $this->db->order_by('p.id_producto', 'DESC'); // Ordenar por ID de forma descendente
        $resultados = $this->db->get();
        return $resultados->result();
    }

    public function getProducto($id) {
        $this->db->where('id_producto', $id);
        $resultado = $this->db->get($this->table);
        return $resultado->row();
    }

    public function save($data) {
        return $this->db->insert($this->table, $data);
    }

    public function update($id_producto, $data) {
        $this->db->where('id_producto', $id_producto);
        return $this->db->update($this->table, $data);
    }

    public function getTotalProductos() {
        $this->db->where('estado', '1');
        $resultados = $this->db->get($this->table);
        return $resultados->num_rows();
    }

    public function getProductosBajoStock() {
        $this->db->where('stock_actual <= stock_minimo');
        $this->db->where('estado', '1');
        $resultados = $this->db->get($this->table);
        return $resultados->result();
    }

    public function getProductosReporte($categoria, $estado_stock) {
        $this->db->select('p.*, c.nombre as categoria');
        $this->db->from($this->table . ' p');
        $this->db->join($this->table_cat . ' c', 'c.id_categoria = p.id_categoria');
        $this->db->where('p.estado', '1');
        
        if($categoria != "") {
            $this->db->where('p.id_categoria', $categoria);
        }
        
        if($estado_stock != "") {
            if($estado_stock == 1) {
                $this->db->where('p.stock_actual > p.stock_minimo');
            } else if($estado_stock == 2) {
                $this->db->where('p.stock_actual <= p.stock_minimo');
                $this->db->where('p.stock_actual > 0');
            } else {
                $this->db->where('p.stock_actual', 0);
            }
        }
        
        $resultados = $this->db->get();
        return $resultados->result();
    }

    public function getProductosByCategoriaAndEstado($categoria, $estado) {
        $this->db->select('p.*, c.nombre as categoria');
        $this->db->from('inv_productos p');
        $this->db->join('inv_categorias c', 'c.id_categoria = p.id_categoria');
        $this->db->where('p.id_categoria', $categoria);
        $this->db->where('p.estado', $estado);
        $resultados = $this->db->get();
        return $resultados->result();
    }

    public function getProductosByCategoria($categoria) {
        $this->db->select('p.*, c.nombre as categoria');
        $this->db->from('inv_productos p');
        $this->db->join('inv_categorias c', 'c.id_categoria = p.id_categoria');
        $this->db->where('p.id_categoria', $categoria);
        $resultados = $this->db->get();
        return $resultados->result();
    }

    public function getProductosByEstado($estado) {
        $this->db->select('p.*, c.nombre as categoria');
        $this->db->from('inv_productos p');
        $this->db->join('inv_categorias c', 'c.id_categoria = p.id_categoria');
        $this->db->where('p.estado', $estado);
        $resultados = $this->db->get();
        return $resultados->result();
    }
    
    /**
     * Obtiene el último producto registrado para generar el código automático
     * @return object|null Último producto o null si no hay productos
     */
    public function getUltimoProducto() {
        $this->db->select("codigo");
        $this->db->from($this->table);
        $this->db->order_by("id_producto", "DESC");
        $this->db->limit(1);
        $resultado = $this->db->get();
        
        if ($resultado->num_rows() > 0) {
            return $resultado->row();
        }
        return null;
    }
    
    /**
     * Verifica si un código de producto ya existe
     * @param string $codigo Código a verificar
     * @return object|boolean Producto si existe, false si no existe
     */
    public function getProductoByCodigo($codigo) {
        $this->db->where("codigo", $codigo);
        $resultado = $this->db->get($this->table);
        
        if ($resultado->num_rows() > 0) {
            return $resultado->row();
        }
        return false;
    }

    public function getValorTotalInventario() {
        $this->db->select('SUM(stock_actual * precio_compra) as valor_total');
        $this->db->from($this->table);
        $this->db->where('estado', '1');
        $resultado = $this->db->get();
        
        if ($resultado->row()->valor_total) {
            return $resultado->row()->valor_total;
        } else {
            return 0; // Retornar 0 si no hay productos o si el valor es NULL
        }
    }

    public function getTotalMovimientosHoy() {
        $this->db->where('DATE(fecha_movimiento)', date('Y-m-d'));
        $this->db->from('inv_movimientos_stock');
        return $this->db->count_all_results();
    }
    
    /**
     * Filtra productos según criterios específicos
     * @param string $categoria ID de categoría
     * @param string $estado Estado del producto (1=activo, 0=inactivo)
     * @param string $stock Condición de stock (bajo, normal)
     * @return array Productos filtrados
     */
    /**
     * Filtra productos según criterios específicos
     * @param string $categoria ID de categoría
     * @param string $estado Estado del producto (1=activo, 0=inactivo)
     * @param string $stock Condición de stock (bajo, normal)
     * @return array Productos filtrados
     */
    public function filtrarProductos($categoria = '', $estado = '', $stock = '') {
        $this->db->select('p.*, c.nombre as categoria');
        $this->db->from($this->table . ' p');
        $this->db->join($this->table_cat . ' c', 'p.id_categoria = c.id_categoria');
        
        // Filtrar por categoría si se especifica
        if (!empty($categoria) && $categoria != 'Todas') {
            $this->db->where('p.id_categoria', $categoria);
        }
        
        // Filtrar por estado si se especifica
        if ($estado !== '' && $estado != 'Todos') {
            $this->db->where('p.estado', $estado);
        }
        
        // Filtrar por stock si se especifica
        if ($stock == 'bajo') {
            $this->db->where('p.stock_actual <= p.stock_minimo');
        } else if ($stock == 'normal') {
            $this->db->where('p.stock_actual > p.stock_minimo');
        } else if ($stock == 'agotado') {
            $this->db->where('p.stock_actual', 0);
        }
        
        $this->db->order_by('p.id_producto', 'DESC');
        $resultados = $this->db->get();
        return $resultados->result();
    }
    
    /**
     * Busca productos por término en nombre o código
     * @param string $termino Término de búsqueda
     * @return array Productos encontrados
     */
    public function buscarProductos($termino) {
        $this->db->select('p.*, c.nombre as categoria');
        $this->db->from($this->table . ' p');
        $this->db->join($this->table_cat . ' c', 'p.id_categoria = c.id_categoria');
        
        // Buscar en nombre o código
        $this->db->group_start();
        $this->db->like('p.nombre', $termino);
        $this->db->or_like('p.codigo', $termino);
        $this->db->group_end();
        
        $this->db->order_by('p.id_producto', 'DESC');
        $resultados = $this->db->get();
        return $resultados->result();
    }

 
    public function getTopProductosSalidas($limit = 5) {
        $this->db->select('p.nombre, COUNT(m.id_movimiento) as movimientos');
        $this->db->from('inv_productos p');
        $this->db->join('inv_movimientos_stock m', 'm.id_producto = p.id_producto');
        $this->db->where('m.tipo_movimiento', 'salida');
        $this->db->group_by('p.id_producto');
        $this->db->order_by('movimientos', 'DESC');
        $this->db->limit($limit);
        return $this->db->get()->result();
    }
}