<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Movimientos_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->table = 'inv_movimientos_stock';
    }

    public function getMovimientos() {
        $this->db->select('m.*, p.nombre as producto, u.nombre as usuario');
        $this->db->from($this->table . ' m');
        $this->db->join('inv_productos p', 'p.id_producto = m.id_producto');
        $this->db->join('usuario u', 'u.codi_usu = m.codi_usu', 'left'); // Agregado LEFT JOIN
        $this->db->order_by('m.fecha_movimiento', 'DESC');
        $resultados = $this->db->get();
        return $resultados->result();
    }
    
    /**
     * Registra una salida de producto en el inventario
     * 
     * @param array $data Datos de la salida (id_producto, cantidad, observaciones)
     * @return boolean True si se registró correctamente, False en caso contrario
     */
    public function registrarSalida($data)
    {
        // Obtener el producto
        $this->db->where('id_producto', $data['id_producto']);
        $producto = $this->db->get('inv_productos')->row();
        
        if (!$producto) {
            return false;
        }
        
        // Verificar que haya suficiente stock
        if ($producto->stock_actual < $data['cantidad']) {
            return false;
        }
        
        // Iniciar transacción
        $this->db->trans_begin();
        
        try {
            // Actualizar el stock del producto
            $nuevo_stock = $producto->stock_actual - $data['cantidad'];
            $this->db->where('id_producto', $data['id_producto']);
            $this->db->update('inv_productos', ['stock_actual' => $nuevo_stock]);
            
            // Registrar el movimiento
            $movimiento = [
                'id_producto' => $data['id_producto'],
                'tipo_movimiento' => 'salida',
                'cantidad' => $data['cantidad'],
                'fecha_movimiento' => date('Y-m-d H:i:s'),
                'codi_usu' => $this->session->userdata('codi_usu'),
                'observaciones' => $data['observaciones']
            ];
            
            $this->db->insert($this->table, $movimiento);
            
            // Confirmar transacción
            if ($this->db->trans_status() === FALSE) {
                $this->db->trans_rollback();
                return false;
            } else {
                $this->db->trans_commit();
                return true;
            }
        } catch (Exception $e) {
            $this->db->trans_rollback();
            return false;
        }
    }
    
    public function save($data) {
        return $this->db->insert($this->table, $data);
    }

    public function getMovimientosReporte($fecha_inicio, $fecha_fin, $tipo) {
        $this->db->select('m.*, p.nombre as producto, u.nombre as usuario');
        $this->db->from($this->table . ' m');
        $this->db->join('inv_productos p', 'p.id_producto = m.id_producto');
        $this->db->join('usuario u', 'u.codi_usu = m.codi_usu', 'left'); // Agregado LEFT JOIN
        $this->db->where('m.fecha_movimiento >=', $fecha_inicio);
        $this->db->where('m.fecha_movimiento <=', $fecha_fin);
        
        if($tipo != "") {
            $this->db->where('m.tipo_movimiento', $tipo);
        }
        
        $this->db->order_by('m.fecha_movimiento', 'DESC');
        $resultados = $this->db->get();
        return $resultados->result();
    }
    
    public function getTotalMovimientosHoy() {
        $this->db->where('DATE(fecha_movimiento)', date('Y-m-d'));
        $this->db->from($this->table);
        return $this->db->count_all_results();
    }
    
    public function getMovimientosFiltrados($fecha_inicio = null, $fecha_fin = null, $tipo = null, $producto = null, $usuario = null) {
        $this->db->select('m.*, p.nombre as producto, u.nombre as usuario');
        $this->db->from($this->table . ' m');
        $this->db->join('inv_productos p', 'p.id_producto = m.id_producto');
        $this->db->join('usuario u', 'u.codi_usu = m.codi_usu', 'left');
        
        // Aplicar filtros si se proporcionan
        if (!empty($fecha_inicio) && !empty($fecha_fin)) {
            $this->db->where('DATE(m.fecha_movimiento) >=', $fecha_inicio);
            $this->db->where('DATE(m.fecha_movimiento) <=', $fecha_fin);
        }
        
        if (!empty($tipo) && $tipo !== 'Todos') {
            $this->db->where('m.tipo_movimiento', $tipo);
        }
        
        if (!empty($producto) && $producto !== 'Todos') {
            $this->db->where('m.id_producto', $producto);
        }
        
        if (!empty($usuario) && $usuario !== 'Todos') {
            $this->db->where('m.codi_usu', $usuario);
        }
        
        $this->db->order_by('m.fecha_movimiento', 'DESC');
        
        return $this->db->get()->result();
    }

        /**
     * Obtiene el último movimiento registrado
     * @return object Último movimiento
     * 
         
     * Obtiene los movimientos de un producto específico
     * @param int $id_producto ID del producto
     * @return array Arreglo con los movimientos
     */
    public function getMovimientosPorProducto($producto_id) {
        $this->db->select('m.*, p.nombre as producto, u.nombre as usuario');
        $this->db->from($this->table . ' m');
        $this->db->join('inv_productos p', 'm.id_producto = p.id_producto');
        $this->db->join('usuario u', 'm.codi_usu = u.codi_usu', 'left');
        $this->db->where('m.id_producto', $producto_id);
        $this->db->order_by('m.fecha_movimiento', 'DESC');
        $resultados = $this->db->get();
        return $resultados->result();
    }

    public function getMovimientosMensuales() {
        $this->db->select("DATE_FORMAT(fecha_movimiento, '%Y-%m') as fecha");
        $this->db->select("SUM(CASE WHEN tipo_movimiento = 'entrada' THEN cantidad ELSE 0 END) as entradas");
        $this->db->select("SUM(CASE WHEN tipo_movimiento = 'salida' THEN cantidad ELSE 0 END) as salidas");
        $this->db->from($this->table);
        $this->db->group_by("DATE_FORMAT(fecha_movimiento, '%Y-%m')");
        $this->db->order_by("fecha", "ASC");
        $this->db->limit(12);
        $result = $this->db->get()->result();
        
        // Si no hay resultados, devolver un array con el mes actual
        if (empty($result)) {
            return [
                (object)[
                    'fecha' => date('Y-m'),
                    'entradas' => 0,
                    'salidas' => 0
                ]
            ];
        }
        
        return $result;
    }
    
}