<?php
/* 
 * Productos.php
 *
 * @author    Desarrollo Medident <**/
defined('BASEPATH') OR exit('No direct script access allowed');

class Productos extends CI_Controller {

    public function __construct() {
        parent::__construct();
        if (!$this->session->userdata("login")) {
            redirect(base_url());
        }
        $this->load->model("Productos_model");
        $this->load->model("Categorias_model");
        $this->load->model("Movimientos_model");
    }

    public function index() {
        // Redirigir al listado de productos en el controlador de inventario
        redirect(base_url('admin/inventario/productos'));
    }

    public function filtrar() {
        // Verificar si es una solicitud AJAX
        if (!$this->input->is_ajax_request()) {
            show_404();
            return;
        }
        
        // Obtener parámetros de filtro
        $categoria = $this->input->post('categoria');
        $estado = $this->input->post('estado');
        $stock = $this->input->post('stock');
        
        // Obtener productos filtrados
        $productos = $this->Productos_model->getProductosFiltrados($categoria, $estado, $stock);
        
        // Preparar datos para la respuesta
        $data = array(
            'productos' => $productos
        );
        
        // Devolver como JSON
        $this->output
            ->set_content_type('application/json')
            ->set_output(json_encode($data));
    }
    
    public function historial($id_producto) {
        // Obtener el producto
        $producto = $this->Productos_model->getProducto($id_producto);
        
        if (!$producto) {
            $this->session->set_flashdata("error", "El producto no existe");
            redirect(base_url('admin/inventario/productos'));
        }
        
        // Obtener los movimientos del producto
        $movimientos = $this->Movimientos_model->getMovimientosPorProducto($id_producto);
        
        $data = array(
            'producto' => $producto,
            'movimientos' => $movimientos
        );
        
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/productos/historial', $data);
        $this->load->view('layouts/footer');
    }
    
    public function activar($id_producto) {
        // Verificar que el producto existe
        $producto = $this->Productos_model->getProducto($id_producto);
        
        if (!$producto) {
            // Si es una solicitud AJAX, devolver JSON
            if ($this->input->is_ajax_request()) {
                $this->output
                    ->set_content_type('application/json')
                    ->set_output(json_encode([
                        'status' => 'error',
                        'message' => 'El producto no existe'
                    ]));
                return;
            }
            
            $this->session->set_flashdata("error", "El producto no existe");
            redirect(base_url('admin/inventario/productos'));
            return;
        }
        
        $data = array(
            'estado' => "1"
        );
        
        if ($this->Productos_model->update($id_producto, $data)) {
            // Registrar en el log de actividades
            $this->registrar_actividad("Activación de producto: ".$producto->nombre);
            
            // Si es una solicitud AJAX, devolver JSON
            if ($this->input->is_ajax_request()) {
                $this->output
                    ->set_content_type('application/json')
                    ->set_output(json_encode([
                        'status' => 'success',
                        'message' => 'El producto ha sido activado correctamente',
                        'id_producto' => $id_producto
                    ]));
                return;
            }
            
            $this->session->set_flashdata("success", "El producto ha sido activado correctamente");
        } else {
            // Si es una solicitud AJAX, devolver JSON
            if ($this->input->is_ajax_request()) {
                $this->output
                    ->set_content_type('application/json')
                    ->set_output(json_encode([
                        'status' => 'error',
                        'message' => 'No se pudo activar el producto'
                    ]));
                return;
            }
            
            $this->session->set_flashdata("error", "No se pudo activar el producto");
        }
        
        redirect(base_url('admin/inventario/productos'));
    }
    
    public function desactivar($id_producto) {
        // Verificar que el producto existe
        $producto = $this->Productos_model->getProducto($id_producto);
        
        if (!$producto) {
            // Si es una solicitud AJAX, devolver JSON
            if ($this->input->is_ajax_request()) {
                $this->output
                    ->set_content_type('application/json')
                    ->set_output(json_encode([
                        'status' => 'error',
                        'message' => 'El producto no existe'
                    ]));
                return;
            }
            
            $this->session->set_flashdata("error", "El producto no existe");
            redirect(base_url('admin/inventario/productos'));
            return;
        }
        
        $data = array(
            'estado' => "0"
        );
        
        if ($this->Productos_model->update($id_producto, $data)) {
            // Si es una solicitud AJAX, devolver JSON
            if ($this->input->is_ajax_request()) {
                $this->output
                    ->set_content_type('application/json')
                    ->set_output(json_encode([
                        'status' => 'success',
                        'message' => 'El producto ha sido desactivado correctamente',
                        'id_producto' => $id_producto
                    ]));
                return;
            }
            
            $this->session->set_flashdata("success", "El producto ha sido desactivado correctamente");
        } else {
            // Si es una solicitud AJAX, devolver JSON
            if ($this->input->is_ajax_request()) {
                $this->output
                    ->set_content_type('application/json')
                    ->set_output(json_encode([
                        'status' => 'error',
                        'message' => 'No se pudo desactivar el producto'
                    ]));
                return;
            }
            
            $this->session->set_flashdata("error", "No se pudo desactivar el producto");
        }
        
        redirect(base_url('admin/inventario/productos'));
    }
    
    // Método auxiliar para registrar actividades
    private function registrar_actividad($descripcion) {
        $data = array(
            'id_usuario' => $this->session->userdata('id'),
            'descripcion' => $descripcion,
            'fecha' => date('Y-m-d H:i:s')
        );
        
        // Si tienes un modelo para registrar actividades, úsalo aquí
        if (method_exists($this, 'Logs_model')) {
            $this->Logs_model->add($data);
        }
    }
    
    public function bajos_stock() {
        // Obtener productos con stock bajo
        $productos = $this->Productos_model->getProductosBajosStock();
        
        $data = array(
            'productos' => $productos
        );
        
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/productos/bajos_stock', $data);
        $this->load->view('layouts/footer');
    }
    
    public function reporte() {
        // Obtener todos los productos
        $productos = $this->Productos_model->getProductos();
        
        // Cargar la librería para exportar a Excel si está disponible
        if (file_exists(APPPATH . 'libraries/PHPExcel.php')) {
            $this->load->library('PHPExcel');
            
            // Crear un nuevo objeto PHPExcel
            $objPHPExcel = new PHPExcel();
            
            // Establecer propiedades del documento
            $objPHPExcel->getProperties()
                ->setCreator("MEDIDENT")
                ->setLastModifiedBy("MEDIDENT")
                ->setTitle("Reporte de Productos")
                ->setSubject("Reporte de Productos")
                ->setDescription("Reporte de Productos del Sistema MEDIDENT")
                ->setKeywords("productos inventario")
                ->setCategory("Reportes");
            
            // Agregar encabezados
            $objPHPExcel->setActiveSheetIndex(0)
                ->setCellValue('A1', 'ID')
                ->setCellValue('B1', 'CÓDIGO')
                ->setCellValue('C1', 'NOMBRE')
                ->setCellValue('D1', 'DESCRIPCIÓN')
                ->setCellValue('E1', 'CATEGORÍA')
                ->setCellValue('F1', 'STOCK ACTUAL')
                ->setCellValue('G1', 'STOCK MÍNIMO')
                ->setCellValue('H1', 'PRECIO COMPRA')
                ->setCellValue('I1', 'PRECIO VENTA')
                ->setCellValue('J1', 'ESTADO');
            
            // Agregar datos
            $fila = 2;
            foreach ($productos as $producto) {
                $objPHPExcel->setActiveSheetIndex(0)
                    ->setCellValue('A'.$fila, $producto->id_producto)
                    ->setCellValue('B'.$fila, $producto->codigo)
                    ->setCellValue('C'.$fila, $producto->nombre)
                    ->setCellValue('D'.$fila, $producto->descripcion)
                    ->setCellValue('E'.$fila, $producto->categoria)
                    ->setCellValue('F'.$fila, $producto->stock_actual)
                    ->setCellValue('G'.$fila, $producto->stock_minimo)
                    ->setCellValue('H'.$fila, $producto->precio_compra)
                    ->setCellValue('I'.$fila, $producto->precio_venta)
                    ->setCellValue('J'.$fila, $producto->estado == 1 ? 'Activo' : 'Inactivo');
                $fila++;
            }
            
            // Ajustar ancho de columnas automáticamente
            foreach(range('A','J') as $columnID) {
                $objPHPExcel->getActiveSheet()->getColumnDimension($columnID)->setAutoSize(true);
            }
            
            // Establecer el nombre de la hoja
            $objPHPExcel->getActiveSheet()->setTitle('Productos');
            
            // Establecer la hoja activa
            $objPHPExcel->setActiveSheetIndex(0);
            
            // Redirigir la salida al navegador
            header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
            header('Content-Disposition: attachment;filename="Reporte_Productos_'.date('Y-m-d').'.xlsx"');
            header('Cache-Control: max-age=0');
            
            $objWriter = PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel2007');
            $objWriter->save('php://output');
            exit;
        } else {
            // Si no está disponible PHPExcel, mostrar un mensaje
            $this->session->set_flashdata("error", "No se pudo generar el reporte. La librería PHPExcel no está disponible.");
            redirect(base_url('admin/inventario/productos'));
        }
    }
}
    