<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Inventario extends CI_Controller {

    public function __construct() {
        parent::__construct();
        if (!$this->session->userdata("login")) {
            redirect(base_url());
        }
        $this->load->model(array(
            'Productos_model',
            'Proveedores_model',
            'Compras_model',
            'Movimientos_model',
            'Categorias_model'
        ));
    }

    public function dashboard() {
        // Cargar el modelo de movimientos si aún no está cargado
        $this->load->model('Movimientos_model');
        
        // Obtener el valor del inventario
        $valor_inventario = $this->Productos_model->getValorTotalInventario();
        
        $data = array(
            'total_productos' => $this->Productos_model->getTotalProductos(),
            'total_proveedores' => $this->Proveedores_model->getTotalProveedores(),
            'productos_bajos' => $this->Productos_model->getProductosBajoStock(),
            'total_movimientos_hoy' => $this->Movimientos_model->getTotalMovimientosHoy(),
            'valor_inventario' => $valor_inventario ? $valor_inventario : 0
        );
        
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/dashboard/index', $data);
        $this->load->view('layouts/footer');
    }

    

    public function productos() {
        $data = array(
            'productos' => $this->Productos_model->getProductos(true), // true para incluir inactivos
            'productos_activos' => $this->Productos_model->getProductos(false) // solo activos para estadísticas
        );
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/productos/list', $data);
        $this->load->view('layouts/footer');
    }




    public function add_producto() {
        // Generamos el código automáticamente
        $ultimo_producto = $this->Productos_model->getUltimoProducto();
        
        if ($ultimo_producto) {
            // Extraer el número del último código (formato P-XXXX)
            $codigo_actual = $ultimo_producto->codigo;
            $numero_actual = intval(substr($codigo_actual, 2)); // Obtener el número después de "P-"
            $nuevo_numero = $numero_actual + 1;
        } else {
            // Si no hay productos, empezamos con 1
            $nuevo_numero = 1;
        }
        
        // Formatear el nuevo código con ceros a la izquierda (P-0001)
        $codigo_generado = 'P-' . str_pad($nuevo_numero, 4, '0', STR_PAD_LEFT);
        
        $data = array(
            'categorias' => $this->Categorias_model->getCategorias(),
            'codigo_generado' => $codigo_generado
        );
        
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/productos/add', $data);
        $this->load->view('layouts/footer');
    }

    public function store_producto() {
        $codigo = $this->input->post("codigo");
        $nombre = $this->input->post("nombre");
        $descripcion = $this->input->post("descripcion");
        $precio_compra = $this->input->post("precio_compra");
        $precio_venta = $this->input->post("precio_venta");
        $stock_inicial = $this->input->post("stock_inicial");
        $stock_minimo = $this->input->post("stock_minimo");
        $categoria = $this->input->post("categoria");
        
        // Verificar que el código no exista ya en la base de datos
        $producto_existente = $this->Productos_model->getProductoByCodigo($codigo);
        
        if ($producto_existente) {
            $this->session->set_flashdata("error", "El código del producto ya existe");
            redirect(base_url('inventario/add_producto'));
        } else {
            // Datos del producto
            $data = array(
                'codigo' => $codigo,
                'nombre' => $nombre,
                'descripcion' => $descripcion,
                'precio_compra' => $precio_compra,
                'precio_venta' => $precio_venta,
                'stock_actual' => $stock_inicial, // Establecer el stock inicial
                'stock_minimo' => $stock_minimo,
                'id_categoria' => $categoria,
                'estado' => "1"
            );

            // Guardar el producto
            if ($this->Productos_model->save($data)) {
                $id_producto = $this->db->insert_id(); // Obtener el ID del producto recién insertado
                
                // Si hay stock inicial, crear un movimiento de ajuste inicial
                if ($stock_inicial > 0) {
                    $movimiento = array(
                        'fecha_movimiento' => date('Y-m-d H:i:s'),
                        'tipo_movimiento' => 'ajuste', // Tipo ajuste para el stock inicial
                        'id_producto' => $id_producto,
                        'cantidad' => $stock_inicial,
                        'codi_usu' => $this->session->userdata('codi_usu'), // Cambiado de 'id' a 'codi_usu'
                        'observaciones' => 'Ajuste Inicial - Creación del producto'
                    );
                    
                    // Guardar el movimiento
                    $this->Movimientos_model->save($movimiento);
                }
                
                redirect(base_url('inventario/productos'));
            } else {
                $this->session->set_flashdata("error", "No se pudo guardar la información");
                redirect(base_url('inventario/add_producto'));
            }
        }
    }

    public function edit_producto($id) {
        $data = array(
            'producto' => $this->Productos_model->getProducto($id),
            'categorias' => $this->Categorias_model->getCategorias()
        );
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/productos/edit', $data);
        $this->load->view('layouts/footer');
    }

    public function update_producto() {
        $id_producto = $this->input->post("id_producto");
        $data = array(
            'codigo' => $this->input->post("codigo"),
            'nombre' => $this->input->post("nombre"),
            'descripcion' => $this->input->post("descripcion"),
            'precio_compra' => $this->input->post("precio_compra"),
            'precio_venta' => $this->input->post("precio_venta"),
            'stock_minimo' => $this->input->post("stock_minimo"),
            'id_categoria' => $this->input->post("categoria")
        );

        if ($this->Productos_model->update($id_producto, $data)) {
            redirect(base_url('admin/inventario/productos'));
        } else {
            $this->session->set_flashdata("error", "No se pudo actualizar la información");
            redirect(base_url('admin/inventario/productos/edit/'.$id_producto));
        }
    }

    // Add this method if it doesn't exist
    public function productos_delete($id_producto) {
        // Instead of deleting, just deactivate
        $data = array(
            'estado' => "0"
        );
        
        if ($this->Productos_model->update($id_producto, $data)) {
            $this->session->set_flashdata("success", "El producto ha sido desactivado correctamente");
        } else {
            $this->session->set_flashdata("error", "No se pudo desactivar el producto");
        }
        
        redirect(base_url('admin/inventario/productos'));
    }

    public function proveedores() {
        $data = array(
            'proveedores' => $this->Proveedores_model->getProveedores()
        );
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/proveedores/list', $data);
        $this->load->view('layouts/footer');
    }

    public function add_proveedor() {
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/proveedores/add');
        $this->load->view('layouts/footer');
    }

    public function store_proveedor() {
        $data = array(
            'ruc' => $this->input->post("ruc"),
            'razon_social' => $this->input->post("razon_social"),
            'nombre_comercial' => $this->input->post("nombre_comercial"),
            'direccion' => $this->input->post("direccion"),
            'telefono' => $this->input->post("telefono"),
            'email' => $this->input->post("email"),
            'contacto_nombre' => $this->input->post("contacto"),
            'telefono_contacto' => $this->input->post("telefono_contacto"),
            'estado' => "1"
        );

        if ($this->Proveedores_model->save($data)) {
            redirect(base_url('admin/inventario/proveedores'));
        } else {
            $this->session->set_flashdata("error", "No se pudo guardar la información");
            redirect(base_url('admin/inventario/proveedores/add'));
        }
    }

    public function edit_proveedor($id) {
        $data = array(
            'proveedor' => $this->Proveedores_model->getProveedor($id)
        );
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/proveedores/edit', $data);
        $this->load->view('layouts/footer');
    }

    public function update_proveedor() {
        $id_proveedor = $this->input->post("id_proveedor");
        $data = array(
            'ruc' => $this->input->post("ruc"),
            'razon_social' => $this->input->post("razon_social"),
            'nombre_comercial' => $this->input->post("nombre_comercial"),
            'direccion' => $this->input->post("direccion"),
            'telefono' => $this->input->post("telefono"),
            'email' => $this->input->post("email"),
            'contacto' => $this->input->post("contacto"),
            'telefono_contacto' => $this->input->post("telefono_contacto")
        );

        if ($this->Proveedores_model->update($id_proveedor, $data)) {
            redirect(base_url('admin/inventario/proveedores'));
        } else {
            $this->session->set_flashdata("error", "No se pudo actualizar la información");
            redirect(base_url('admin/inventario/proveedores/edit/'.$id_proveedor));
        }
    }

    public function delete_proveedor($id) {
        $data = array(
            'estado' => "0"
        );
        $this->Proveedores_model->update($id, $data);
        redirect(base_url('admin/inventario/proveedores'));
    }

    public function compras() {
        $data = array(
            'compras' => $this->Compras_model->getCompras()
        );
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/compras/list', $data);
        $this->load->view('layouts/footer');
    }

    public function add_compra() {
        $data = array(
            'productos' => $this->Productos_model->getProductos(),
            'proveedores' => $this->Proveedores_model->getProveedores()
        );
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/compras/add', $data);
        $this->load->view('layouts/footer');
    }

    public function store_compra() {
        $fecha = $this->input->post("fecha");
        $proveedor = $this->input->post("proveedor");
        $comprobante = $this->input->post("comprobante");
        $productos = $this->input->post("productos");
        $cantidades = $this->input->post("cantidades");
        $precios = $this->input->post("precios");
        $importes = $this->input->post("importes");
        $total = $this->input->post("total");

        $data = array(
            'fecha_compra' => $fecha,
            'id_proveedor' => $proveedor,
            'factura_nro' => $comprobante,
            'total' => $total,
            'estado' => "1"
        );

        if ($this->Compras_model->save($data)) {
            $compra_id = $this->Compras_model->lastID();
            $this->save_detalle_compra($compra_id, $productos, $cantidades, $precios, $importes);   
            $this->actualizar_stock($productos, $cantidades);
            
            // Registrar movimientos de entrada para cada producto comprado
            $this->registrar_movimientos_compra($compra_id, $productos, $cantidades, $comprobante);
            
            redirect(base_url('admin/inventario/compras'));
        } else {
            $this->session->set_flashdata("error", "No se pudo guardar la información");
            redirect(base_url('admin/inventario/compras/add'));
        }
    }

    // Nuevo método para registrar los movimientos de compra
    protected function registrar_movimientos_compra($compra_id, $productos, $cantidades, $comprobante) {
        for ($i=0; $i < count($productos); $i++) {
            $producto_info = $this->Productos_model->getProducto($productos[$i]);
            
            $movimiento = array(
                'fecha_movimiento' => date('Y-m-d H:i:s'),
                'tipo_movimiento' => 'entrada',
                'id_producto' => $productos[$i],
                'cantidad' => $cantidades[$i],
                'codi_usu' => $this->session->userdata('codi_usu'),
                'observaciones' => 'Entrada por compra #' . $compra_id . ' - Comprobante: ' . $comprobante
            );
            
            // Guardar el movimiento
            $this->Movimientos_model->save($movimiento);
        }
    }

    protected function save_detalle_compra($compra_id, $productos, $cantidades, $precios, $importes) {
        for ($i=0; $i < count($productos); $i++) { 
            $data = array(
                'id_compra' => $compra_id,
                'id_producto' => $productos[$i],
                'cantidad' => $cantidades[$i],
                'precio_unitario' => $precios[$i],
                'subtotal' => $importes[$i]
            );
            $this->Compras_model->save_detalle($data);
        }
    }

    protected function actualizar_stock($productos, $cantidades) {
        for ($i=0; $i < count($productos); $i++) { 
            $productoActual = $this->Productos_model->getProducto($productos[$i]);
            $data = array(
                'stock_actual' => $productoActual->stock_actual + $cantidades[$i]
            );
            $this->Productos_model->update($productos[$i], $data);
        }
    }

    public function view_compra() {
        $id_compra = $this->input->post("id");
        $data = array(
            'compra' => $this->Compras_model->getCompra($id_compra),
            'detalles' => $this->Compras_model->getDetalle($id_compra)
        );
        $this->load->view("admin/inventario/compras/view", $data);
    }

    public function anular_compra($id) {
        // Obtener los detalles de la compra antes de anularla
        $detalles = $this->Compras_model->getDetalle($id);
        
        // Revertir el stock de los productos
        foreach ($detalles as $detalle) {
            $producto = $this->Productos_model->getProducto($detalle->id_producto);
            $nuevo_stock = $producto->stock_actual - $detalle->cantidad;
            
            $data_producto = array(
                'stock_actual' => $nuevo_stock
            );
            
            $this->Productos_model->update($detalle->id_producto, $data_producto);
            
            // Registrar movimiento de ajuste por anulación
            $movimiento = array(
                'fecha_movimiento' => date('Y-m-d H:i:s'),
                'tipo_movimiento' => 'salida',
                'id_producto' => $detalle->id_producto,
                'cantidad' => $detalle->cantidad,
                'codi_usu' => $this->session->userdata('codi_usu'),
                'observaciones' => 'Anulación de compra #' . $id
            );
            
            $this->Movimientos_model->save($movimiento);
        }
        
        // Actualizar el estado de la compra a cancelada
        $data = array(
            'estado' => "cancelada"
        );
        
        if ($this->Compras_model->update($id, $data)) {
            $this->session->set_flashdata("success", "La compra ha sido anulada correctamente");
        } else {
            $this->session->set_flashdata("error", "No se pudo anular la compra");
        }
        
        redirect(base_url('admin/inventario/compras'));
    }

    public function recibir_compra($id) {
        try {
            // Iniciar transacción
            $this->db->trans_begin();
            
            $data = array(
                'estado' => "recibida"
            );
            
            // Actualizar el estado de la compra
            if ($this->Compras_model->update($id, $data)) {
                // Confirmar transacción
                $this->db->trans_commit();
                $this->session->set_flashdata("success", "La compra ha sido marcada como recibida correctamente");
            } else {
                // Revertir en caso de error
                $this->db->trans_rollback();
                $this->session->set_flashdata("error", "No se pudo actualizar el estado de la compra");
            }
        } catch (Exception $e) {
            // Manejar cualquier error
            $this->db->trans_rollback();
            $this->session->set_flashdata("error", "Error al procesar la solicitud: " . $e->getMessage());
        }
        
        redirect(base_url('admin/inventario/compras'));
    }

    public function movimientos() {
        // Asegúrate de que el modelo esté cargado
        $this->load->model('Movimientos_model');
        
        // Calcular estadísticas para el dashboard
        $estadisticas = array(
            'total_entradas' => 0,
            'total_salidas' => 0,
            'stock_afectado' => 0,
            'producto_mas_movido' => 'N/A'
        );
        
        $movimientos = $this->Movimientos_model->getMovimientos();
        $productos_movimientos = array();
        
        if (!empty($movimientos)) {
            foreach ($movimientos as $movimiento) {
                // Contar entradas y salidas
                if ($movimiento->tipo_movimiento == 'entrada') {
                    $estadisticas['total_entradas']++;
                    $estadisticas['stock_afectado'] += $movimiento->cantidad;
                } elseif ($movimiento->tipo_movimiento == 'salida') {
                    $estadisticas['total_salidas']++;
                    $estadisticas['stock_afectado'] += $movimiento->cantidad;
                } elseif ($movimiento->tipo_movimiento == 'ajuste') {
                    $estadisticas['stock_afectado'] += abs($movimiento->cantidad);
                }
                
                // Contar movimientos por producto
                if (isset($movimiento->producto)) {
                    if (!isset($productos_movimientos[$movimiento->producto])) {
                        $productos_movimientos[$movimiento->producto] = 0;
                    }
                    $productos_movimientos[$movimiento->producto]++;
                }
            }
            
            // Encontrar el producto más movido
            if (!empty($productos_movimientos)) {
                arsort($productos_movimientos);
                $estadisticas['producto_mas_movido'] = key($productos_movimientos);
            }
        }
        
        $data = array(
            'movimientos' => $movimientos,
            'estadisticas' => $estadisticas,
            'productos' => $this->Productos_model->getProductos(),
            'usuarios' => $this->db->get('usuario')->result() // Correcto: 'usuario' en singular
        );
        
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/movimientos/list', $data);
        $this->load->view('layouts/footer');
    }

    // Método para filtrar movimientos vía AJAX
    public function filtrar_movimientos() {
        // Verificar si es una solicitud AJAX
        if (!$this->input->is_ajax_request()) {
            show_404();
            return;
        }
        
        try {
            // Obtener parámetros de filtro
            $fecha_inicio = $this->input->post('fecha_inicio');
            $fecha_fin = $this->input->post('fecha_fin');
            $tipo = $this->input->post('tipo_movimiento');
            $producto = $this->input->post('producto');
            $usuario = $this->input->post('usuario');
            
            // Depuración
            log_message('debug', 'Filtros: ' . json_encode([
                'fecha_inicio' => $fecha_inicio,
                'fecha_fin' => $fecha_fin,
                'tipo' => $tipo,
                'producto' => $producto,
                'usuario' => $usuario
            ]));
            
            // Asegurarse de que el modelo esté cargado
            $this->load->model('Movimientos_model');
            
            // Obtener movimientos filtrados
            $movimientos = $this->Movimientos_model->getMovimientosFiltrados(
                $fecha_inicio, 
                $fecha_fin, 
                $tipo, 
                $producto, 
                $usuario
            );
            
            // Calcular estadísticas basadas en los movimientos filtrados
            $estadisticas = array(
                'total_entradas' => 0,
                'total_salidas' => 0,
                'stock_afectado' => 0,
                'producto_mas_movido' => 'N/A'
            );
            
            $productos_movimientos = array();
            
            if (!empty($movimientos)) {
                foreach ($movimientos as $movimiento) {
                    // Contar entradas y salidas
                    if ($movimiento->tipo_movimiento == 'entrada') {
                        $estadisticas['total_entradas']++;
                        $estadisticas['stock_afectado'] += $movimiento->cantidad;
                    } elseif ($movimiento->tipo_movimiento == 'salida') {
                        $estadisticas['total_salidas']++;
                        $estadisticas['stock_afectado'] += $movimiento->cantidad;
                    } elseif ($movimiento->tipo_movimiento == 'ajuste') {
                        $estadisticas['stock_afectado'] += abs($movimiento->cantidad);
                    }
                    
                    // Contar movimientos por producto
                    if (isset($movimiento->producto)) {
                        if (!isset($productos_movimientos[$movimiento->producto])) {
                            $productos_movimientos[$movimiento->producto] = 0;
                        }
                        $productos_movimientos[$movimiento->producto]++;
                    }
                }
                
                // Encontrar el producto más movido
                if (!empty($productos_movimientos)) {
                    arsort($productos_movimientos);
                    $estadisticas['producto_mas_movido'] = key($productos_movimientos);
                }
            }
            
            // Preparar datos para la respuesta
            $data = array(
                'movimientos' => $movimientos,
                'estadisticas' => $estadisticas
            );
            
            // Devolver como JSON
            $this->output
                ->set_content_type('application/json')
                ->set_output(json_encode($data));
                
        } catch (Exception $e) {
            // Registrar el error
            log_message('error', 'Error en filtrar_movimientos: ' . $e->getMessage());
            
            // Devolver error
            $this->output
                ->set_status_header(500)
                ->set_content_type('application/json')
                ->set_output(json_encode(['error' => 'Error al filtrar los datos: ' . $e->getMessage()]));
        }
    }

    public function add_movimiento() {
        $data = array(
            'productos' => $this->Productos_model->getProductos()
        );
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/movimientos/add', $data);
        $this->load->view('layouts/footer');
    }

    public function store_movimiento() {
        $fecha = date('Y-m-d H:i:s'); // Usamos la fecha actual
        $tipo = $this->input->post("tipo_movimiento");
        $producto = $this->input->post("producto");
        $cantidad = $this->input->post("cantidad");
        $observacion = $this->input->post("observaciones");
        
        $data = array(
            'fecha_movimiento' => $fecha,
            'tipo_movimiento' => $tipo,
            'id_producto' => $producto,
            'cantidad' => $cantidad,
            'codi_usu' => $this->session->userdata("codi_usu"), // Asegúrate de usar codi_usu
            'observaciones' => $observacion
        );        
    
        if ($this->Movimientos_model->save($data)) {
            $this->actualizar_stock_movimiento($producto, $cantidad, $tipo);
            redirect(base_url('inventario/movimientos'));
        } else {
            $this->session->set_flashdata("error", "No se pudo registrar el movimiento");
            redirect(base_url('admin/inventario/movimientos/add'));
        }
    }

    protected function actualizar_stock_movimiento($producto_id, $cantidad, $tipo) {
        $producto = $this->Productos_model->getProducto($producto_id);
        if ($tipo == "entrada") {
            $nuevo_stock = $producto->stock_actual + $cantidad;
        } else if ($tipo == "salida") {
            $nuevo_stock = $producto->stock_actual - $cantidad;
        } else if ($tipo == "ajuste") {
            // Para ajustes, simplemente establecemos el stock al valor especificado
            $nuevo_stock = $cantidad;
        }
        
        $data = array(
            'stock_actual' => $nuevo_stock
        );
        $this->Productos_model->update($producto_id, $data);
    }

    public function reportes() {
        $data = array(
            'categorias' => $this->Categorias_model->getCategorias()
        );
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/reportes/index', $data);
        $this->load->view('layouts/footer');
    }

    public function reportes_stock() {
        $categoria = $this->input->post("categoria");
        $estado = $this->input->post("estado");
        
        // Obtener productos según los filtros
        if ($categoria == "Todas" && $estado == "Todos") {
            $productos = $this->Productos_model->getProductos();
        } else if ($categoria == "Todas" && $estado != "Todos") {
            $productos = $this->Productos_model->getProductosByEstado($estado);
        } else if ($categoria != "Todas" && $estado == "Todos") {
            $productos = $this->Productos_model->getProductosByCategoria($categoria);
        } else {
            $productos = $this->Productos_model->getProductosByCategoriaAndEstado($categoria, $estado);
        }
        
        // Pasar los datos a la vista
        $data = array(
            'productos' => $productos
        );
        
        // Cargar la vista
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/reportes/reports_stock', $data);
        $this->load->view('layouts/footer');
    }

    public function reporte_movimientos() {
        $fechas = $this->input->post("fechas");
        $tipo = $this->input->post("tipo");
        
        $array_fechas = explode(" - ", $fechas);
        $fecha_inicio = $array_fechas[0];
        $fecha_fin = $array_fechas[1];
        
        $data = array(
            'movimientos' => $this->Movimientos_model->getMovimientosReporte($fecha_inicio, $fecha_fin, $tipo)
        );
        
        $this->load->view('admin/inventario/reportes/reporte_movimientos', $data);
    }

    public function categorias() {
        $data = array(
            'categorias' => $this->Categorias_model->getCategorias()
        );
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/categorias/list', $data);
        $this->load->view('layouts/footer');
    }

    public function add_categoria() {
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/categorias/add');
        $this->load->view('layouts/footer');
    }

    public function store_categoria() {
        $data = array(
            'nombre' => $this->input->post("nombre"),
            'descripcion' => $this->input->post("descripcion"),
            'estado' => "1"
        );

        if ($this->Categorias_model->save($data)) {
            redirect(base_url('inventario/categorias'));
        } else {
            $this->session->set_flashdata("error", "No se pudo guardar la información");
            redirect(base_url('inventario/categorias/add'));
        }
    }

    public function edit_categoria($id) {
        $data = array(
            'categoria' => $this->Categorias_model->getCategoria($id)
        );
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/categorias/edit', $data);
        $this->load->view('layouts/footer');
    }

    public function update_categoria($id_categoria) {
        $data = array(
            'nombre' => $this->input->post("nombre"),
            'descripcion' => $this->input->post("descripcion")
        );

        if ($this->Categorias_model->update($id_categoria, $data)) {
            redirect(base_url('inventario/categorias'));
        } else {
            $this->session->set_flashdata("error", "No se pudo actualizar la información");
            redirect(base_url('inventario/categorias/edit/'.$id_categoria));
        }
    }

    public function delete_categoria($id) {
        $data = array(
            'estado' => "0"
        );
        $this->Categorias_model->update($id, $data);
        redirect(base_url('inventario/categorias'));
    }

    /**
     * Registra una salida de producto
     */
    public function store_salida()
    {
        // Validar datos de entrada
        $this->form_validation->set_rules('id_producto', 'Producto', 'required|numeric');
        $this->form_validation->set_rules('cantidad', 'Cantidad', 'required|numeric|greater_than[0]');
        
        if ($this->form_validation->run() == FALSE) {
            $this->session->set_flashdata('error', 'Error en los datos ingresados');
            redirect(base_url('admin/inventario/productos'));
            return;
        }
        
        // Preparar datos para el registro
        $data = [
            'id_producto' => $this->input->post('id_producto'),
            'cantidad' => $this->input->post('cantidad'),
            'observaciones' => $this->input->post('observaciones')
        ];
        
        // Registrar la salida
        if ($this->Movimientos_model->registrarSalida($data)) {
            $this->session->set_flashdata('success', 'Salida registrada correctamente');
        } else {
            $this->session->set_flashdata('error', 'No se pudo registrar la salida');
        }
        
        redirect(base_url('admin/inventario/productos'));
    }

    public function historial($id_producto) {
        // Obtener datos del producto
        $this->load->model('Productos_model');
        $this->load->model('Movimientos_model');
        
        $producto = $this->Productos_model->getProducto($id_producto);
        
        if (!$producto) {
            $this->session->set_flashdata('error', 'El producto no existe');
            redirect(base_url('admin/inventario/productos'));
        }
        
        // Obtener movimientos del producto
        $movimientos = $this->Movimientos_model->getMovimientosPorProducto($id_producto);
        
        // Preparar datos para la vista
        $data = array(
            'producto' => $producto,
            'movimientos' => $movimientos
        );
        
        // Cargar vistas
        $this->load->view('layouts/header');
        $this->load->view('layouts/aside');
        $this->load->view('admin/inventario/productos/historial', $data);
        $this->load->view('layouts/footer');
    }
}
