<?php
// routes/web.php

use App\Http\Controllers\Admin\DashboardController as AdminDashboardController;
use App\Http\Controllers\Admin\UserController;
use App\Http\Controllers\Operador\DashboardController as OperadorDashboardController;
use App\Http\Controllers\Proveedor\DashboardController as ProveedorDashboardController;
use App\Http\Controllers\EmpresaController;
use App\Http\Controllers\AnydeskTerminalController;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\NotificacionController;
use App\Http\Controllers\ProfileController;
use App\Http\Controllers\PublicoController;
use App\Http\Controllers\ReporteController;
use App\Http\Controllers\TareaController;
use App\Models\Auditoria;
use Illuminate\Support\Facades\Artisan;  // Added for storage:link route
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;

// Rutas de autenticación
Auth::routes();

// Rutas Públicas
Route::get('/empresas/credenciales', [EmpresaController::class, 'credenciales'])->name('empresas.credenciales');
Route::get('/consulta-ruc', [PublicoController::class, 'index'])->name('publico.index');
Route::post('/consulta-ruc', [PublicoController::class, 'buscar'])
    ->name('publico.buscar')
    ->middleware('throttle:5,1');  // Limit to 5 attempts per minute

// Ruta principal
Route::get('/', function () {
    return redirect()->route('login');
});

// Rutas autenticadas
Route::middleware('auth')->group(function () {
    Route::get('/home', [HomeController::class, 'index'])->name('home');

    // Rutas de Perfil
    Route::get('/perfil', [ProfileController::class, 'edit'])->name('profile.edit');
    Route::patch('/perfil', [ProfileController::class, 'update'])->name('profile.update');

    // Rutas específicas de notificaciones - IMPORTANTE: orden correcto (antes del resource)
    Route::patch('notificaciones/mark-all-read', [NotificacionController::class, 'markAllAsRead'])->name('notificaciones.mark-all-read');
    Route::get('notificaciones/count', [NotificacionController::class, 'getUnreadCount'])->name('notificaciones.count');
    Route::delete('notificaciones/clear-all', [NotificacionController::class, 'clearAll'])->name('notificaciones.clear-all');
    Route::patch('notificaciones/{notificacion}/read', [NotificacionController::class, 'markAsRead'])->name('notificaciones.mark-read');

    // Rutas de notificaciones (resource al final)
    Route::resource('notificaciones', NotificacionController::class)->except(['edit', 'update']);

    // Rutas de empresas
    Route::resource('empresas', EmpresaController::class);
    Route::get('/empresas/{empresa}/modal-content', [EmpresaController::class, 'modalContent'])->name('empresas.modal-content');
    Route::get('/empresas/{empresa}/modal-tareas', [EmpresaController::class, 'modalTareasContent'])->name('empresas.modal-tareas');
    Route::get('/empresas/{empresa}/modal-add-tasks', [EmpresaController::class, 'modalAddTasksContent'])->name('empresas.modal-add-tasks');
    // Nuevas rutas para agregar tareas
    Route::get('/empresas/{empresa}/add-tareas', [EmpresaController::class, 'addTareas'])->name('empresas.add-tareas');
    Route::post('/empresas/{empresa}/add-tareas', [EmpresaController::class, 'storeTareas'])->name('empresas.store-tareas');
    // Agregar otros RUC a una empresa
    Route::get('/empresas/{empresa}/modal-rucs', [EmpresaController::class, 'modalRucs'])->name('empresas.modal-rucs');
    Route::get('/empresas/{empresa}/modal-ruc-form', [EmpresaController::class, 'modalRucForm'])->name('empresas.modal-ruc-form');
    Route::post('/empresas/{empresa}/rucs', [EmpresaController::class, 'storeRuc'])->name('empresas.rucs.store');
    // Edit/Delete RUCs
    Route::get('/empresas/rucs/{empresaRuc}/edit-modal', [EmpresaController::class, 'modalRucEditForm'])->name('empresas.rucs.modal-edit');
    Route::put('/empresas/rucs/{empresaRuc}', [EmpresaController::class, 'updateRuc'])->name('empresas.rucs.update');
    Route::delete('/empresas/rucs/{empresaRuc}', [EmpresaController::class, 'destroyRuc'])->name('empresas.rucs.destroy');

    Route::patch('/empresas/{empresa}/approve', [EmpresaController::class, 'approve'])->name('empresas.approve');
    Route::patch('/empresas/{empresa}/reject', [EmpresaController::class, 'reject'])->name('empresas.reject');
    Route::patch('/empresas/{empresa}/update-status', [EmpresaController::class, 'updateStatus'])->name('empresas.update-status');

    // Rutas de Reportes (Proveedor y Administrador)
    Route::get('/reportes', [ReporteController::class, 'index'])->name('reportes.index');
    Route::post('/reportes/exportar', [ReporteController::class, 'exportar'])->name('reportes.exportar');

    // Rutas de tareas
    Route::get('/tareas/kanban', [TareaController::class, 'kanban'])->name('tareas.kanban');
    Route::post('/tareas/kanban/update-order', [TareaController::class, 'updateOrder'])->name('tareas.kanban.update-order');  // Optional: for ordering within column
    Route::resource('tareas', TareaController::class);
    Route::get('/tareas/{tarea}/modal-content', [TareaController::class, 'modalContent'])->name('tareas.modal-content');
    Route::get('/tareas/{tarea}/modal-edit-content', [TareaController::class, 'modalEditContent'])->name('tareas.modal-edit-content');
    Route::patch('/tareas/{tarea}/status', [TareaController::class, 'updateStatus'])->name('tareas.update-status');
    Route::post('/tareas/{tarea}/comment', [TareaController::class, 'addComment'])->name('tareas.comment');
    Route::post('/tareas/{tarea}/evidence', [TareaController::class, 'uploadEvidence'])->name('tareas.upload-evidence');
    Route::delete('/evidencias/{evidencia}', [TareaController::class, 'deleteEvidence'])->name('evidencias.destroy');
    Route::get('/evidencias/{evidencia}/download', [TareaController::class, 'downloadEvidence'])->name('tareas.download-evidence');

    // Rutas de materiales de apoyo
    Route::post('/tareas/{tarea}/material', [TareaController::class, 'uploadMaterial'])->name('tareas.upload-material');
    Route::delete('/materiales/{material}', [TareaController::class, 'deleteMaterial'])->name('materiales.destroy');
    Route::get('/materiales/{material}/download', [TareaController::class, 'downloadMaterial'])->name('materiales.download');

    // Rutas específicas por rol
    Route::middleware('role:administrador')->prefix('admin')->name('admin.')->group(function () {
        Route::get('/dashboard', [AdminDashboardController::class, 'index'])->name('dashboard');

        // Gestión de Terminales Anydesk
        Route::resource('anydesk', AnydeskTerminalController::class)->names('anydesk');

        // Gestión de Usuarios
        Route::patch('/users/{user}/toggle-status', [UserController::class, 'toggleStatus'])->name('users.toggle-status');
        Route::resource('users', UserController::class)->except(['create', 'edit']);

        // Gestión de API Keys
        Route::get('/api-keys', [App\Http\Controllers\Admin\ApiKeyController::class, 'index'])->name('api-keys.index');
        Route::post('/api-keys/regenerate', [App\Http\Controllers\Admin\ApiKeyController::class, 'regenerate'])->name('api-keys.regenerate');
    });

    Route::middleware('role:proveedor')->prefix('proveedor')->name('proveedor.')->group(function () {
        Route::get('/dashboard', [ProveedorDashboardController::class, 'index'])->name('dashboard');
    });

    Route::middleware('role:operador')->prefix('operador')->name('operador.')->group(function () {
        Route::get('/dashboard', [OperadorDashboardController::class, 'index'])->name('dashboard');
    });
});

// Route to fix storage link on hosting
Route::get('/storage-link', function () {
    $targetFolder = storage_path('app/public');
    $linkFolder = public_path('storage');
    $output = "Iniciando proceso...\n";

    try {
        // 1. Verificar carpetas actuales
        $output .= "Target (Real): $targetFolder \n";
        $output .= "Link (Public): $linkFolder \n";

        if (file_exists($linkFolder)) {
            $output .= "La carpeta public/storage EXISTE.\n";
            if (is_link($linkFolder)) {
                $output .= "Es un LINK. Borrando...\n";
                unlink($linkFolder);
            } else {
                $output .= "Es una CARPETA FÍSICA. Renombrando a backup...\n";
                rename($linkFolder, public_path('storage_backup_' . time()));
            }
        } else {
            $output .= "La carpeta public/storage NO existe.\n";
        }

        // 2. Crear el link manualmente con symlink (a veces Artisan falla en hostings raros)
        // Intentar primero con Artisan
        try {
            Artisan::call('storage:link');
            $output .= "Artisan storage:link ejecutado.\n";
        } catch (\Exception $e) {
            $output .= 'Fallo Artisan: ' . $e->getMessage() . "\n";
            // Intento manual con PHP nativo
            if (symlink($targetFolder, $linkFolder)) {
                $output .= "Symlink creado manualmente con PHP.\n";
            } else {
                $output .= "Fallo symlink manual.\n";
            }
        }

        // 3. Verificación final
        if (is_link($linkFolder)) {
            $output .= "ÉXITO: public/storage ahora es un LINK correcto.\n";
        } else {
            $output .= "ERROR FINAL: public/storage no es un link.\n";
        }

        return nl2br($output);
    } catch (\Exception $e) {
        return 'Error Fatal: ' . $e->getMessage();
    }
});
