@extends('layouts.app')

@section('title', 'Tablero Kanban')

@section('styles')
<style>
    .kanban-col {
        min-height: 600px;
        background-color: #f4f6f9;
        border-radius: 4px;
    }
    .kanban-card {
        cursor: grab;
        transition: transform 0.2s, box-shadow 0.2s;
    }
    .kanban-card:active {
        cursor: grabbing;
    }
    .kanban-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 6px rgba(0,0,0,0.1);
    }
    .ghost-card {
        background: #e9ecef;
        border: 2px dashed #adb5bd;
        opacity: 0.5;
    }
</style>
@endsection

@section('content')
    <div class="container-fluid">
        <div class="row">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h3 class="mb-0"><i class="fas fa-columns me-2"></i> Tablero de Tareas</h3>
                        <p class="text-muted mb-0">Gestión visual del flujo de trabajo</p>
                    </div>
                    <div>
                        <a href="{{ route('tareas.index') }}" class="btn btn-outline-secondary">
                            <i class="fas fa-list"></i> Ver Lista
                        </a>
                        <a href="{{ route('tareas.create') }}" class="btn btn-primary">
                            <i class="fas fa-plus"></i> Nueva Tarea
                        </a>
                    </div>
                </div>

                <!-- Filtros -->
                <div class="card mb-4">
                    <div class="card-body py-3">
                        <form action="{{ route('tareas.kanban') }}" method="GET" class="row align-items-end">
                            <div class="col-md-4">
                                <label class="form-label small text-muted">Empresa</label>
                                <select name="empresa_id" class="form-select form-select-sm" onchange="this.form.submit()">
                                    <option value="">Todas las empresas</option>
                                    @foreach($empresas as $empresa)
                                        <option value="{{ $empresa->id }}" {{ request('empresa_id') == $empresa->id ? 'selected' : '' }}>
                                            {{ $empresa->nombre_comercial ?? $empresa->razon_social }}
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            <div class="col-md-4">
                                <label class="form-label small text-muted">Buscar</label>
                                <div class="input-group input-group-sm">
                                    <input type="text" name="search" class="form-control"
                                        placeholder="Título o descripción..." value="{{ request('search') }}">
                                    <button class="btn btn-outline-secondary" type="submit">
                                        <i class="fas fa-search"></i>
                                    </button>
                                </div>
                            </div>
                            <div class="col-md-4 text-end">
                                <a href="{{ route('tareas.kanban') }}" class="btn btn-sm btn-link text-decoration-none text-muted">
                                    Limpiar filtros
                                </a>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Kanban Board -->
                <div class="row">
                    <!-- Pendiente -->
                    <div class="col-md-4">
                        <div class="card h-100 shadow-sm">
                            <div class="card-header bg-warning text-dark font-weight-bold d-flex justify-content-between align-items-center">
                                <span>PENDIENTE</span>
                                <span class="badge badge-light badge-pill">{{ $kanbanData['pendiente']->count() }}</span>
                            </div>
                            <div class="card-body p-2 kanban-col" id="col-pendiente" data-status="pendiente">
                                @forelse($kanbanData['pendiente'] as $tarea)
                                    @include('tareas.partials.card', ['tarea' => $tarea])
                                @empty
                                    <div class="text-center text-muted p-3 empty-msg">No hay tareas</div>
                                @endforelse
                            </div>
                        </div>
                    </div>

                    <!-- En Progreso -->
                    <div class="col-md-4">
                        <div class="card h-100 shadow-sm">
                            <div class="card-header bg-primary text-white font-weight-bold d-flex justify-content-between align-items-center">
                                <span>EN PROGRESO</span>
                                <span class="badge badge-light badge-pill text-dark">{{ $kanbanData['en_progreso']->count() }}</span>
                            </div>
                            <div class="card-body p-2 kanban-col" id="col-en_progreso" data-status="en_progreso">
                                @forelse($kanbanData['en_progreso'] as $tarea)
                                    @include('tareas.partials.card', ['tarea' => $tarea])
                                @empty
                                    <div class="text-center text-muted p-3 empty-msg">No hay tareas</div>
                                @endforelse
                            </div>
                        </div>
                    </div>

                    <!-- Finalizado -->
                    <div class="col-md-4">
                        <div class="card h-100 shadow-sm">
                            <div class="card-header bg-success text-white font-weight-bold d-flex justify-content-between align-items-center">
                                <span>FINALIZADO</span>
                                <span class="badge badge-light badge-pill text-dark">{{ $kanbanData['finalizado']->count() }}</span>
                            </div>
                            <div class="card-body p-2 kanban-col" id="col-finalizado" data-status="finalizado">
                                @forelse($kanbanData['finalizado'] as $tarea)
                                    @include('tareas.partials.card', ['tarea' => $tarea])
                                @empty
                                    <div class="text-center text-muted p-3 empty-msg">No hay tareas</div>
                                @endforelse
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@section('scripts')
    <!-- SortableJS -->
    <script src="https://cdn.jsdelivr.net/npm/sortablejs@latest/Sortable.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

    <script>
        document.addEventListener('DOMContentLoaded', function () {
            const columns = document.querySelectorAll('.kanban-col');

            columns.forEach(col => {
                new Sortable(col, {
                    group: 'kanban', // Allow dragging between columns
                    animation: 150,
                    ghostClass: 'ghost-card',
                    draggable: '.kanban-card',
                    onEnd: function (evt) {
                        const itemEl = evt.item;
                        const newStatus = evt.to.getAttribute('data-status');
                        const oldStatus = evt.from.getAttribute('data-status');
                        const tareaId = itemEl.getAttribute('data-id');

                        if (newStatus === oldStatus) return; // No change

                        // If moving to Finalizado, ask for observations
                        if (newStatus === 'finalizado') {
                            confirmarFinalizacion(tareaId, oldStatus, evt);
                        } else {
                            // Update status directly
                            updateStatus(tareaId, newStatus, evt);
                        }
                    }
                });
            });
        });

        function updateStatus(tareaId, status, evt, observaciones = null) {
            const url = `{{ url('tareas') }}/${tareaId}/status`;
            
            const data = {
                _method: 'PATCH',
                _token: '{{ csrf_token() }}',
                estado: status
            };

            if (observaciones) {
                data.observaciones_finales = observaciones;
            }

            // Return promise for better handling
            return $.ajax({
                url: url,
                type: 'POST',
                dataType: 'json',
                data: data,
                success: function(response) {
                    const Toast = Swal.mixin({
                        toast: true,
                        position: 'top-end',
                        showConfirmButton: false,
                        timer: 3000
                    });
                    Toast.fire({
                        icon: 'success',
                        title: response.message
                    });
                },
                error: function(xhr) {
                    console.error(xhr);
                    Swal.fire('Error', 'No se pudo actualizar el estado', 'error');
                    // Revert drag
                    if(evt) {
                        // SortableJS doesn't inherently revert, rely on page reload or manual DOM manipulation if critical
                        // Simple fallback: reload to ensure state consistency
                        // location.reload();
                    }
                }
            });
        }

        function confirmarFinalizacion(tareaId, oldStatus, evt) {
            Swal.fire({
                title: 'Finalizar Tarea',
                input: 'textarea',
                inputLabel: 'Observaciones Finales (Solución)',
                inputPlaceholder: 'Describa la solución...',
                inputAttributes: {
                    'aria-label': 'Describa la solución'
                },
                showCancelButton: true,
                confirmButtonText: 'Finalizar',
                cancelButtonText: 'Cancelar',
                showLoaderOnConfirm: true,
                preConfirm: (observacion) => {
                    // Make it uppercase
                    return observacion ? observacion.toUpperCase() : '';
                },
                allowOutsideClick: () => !Swal.isLoading()
            }).then((result) => {
                if (result.isConfirmed) {
                    updateStatus(tareaId, 'finalizado', evt, result.value);
                } else {
                    // User cancelled, move card back
                    // This is tricky with SortableJS generic API, reload is safest
                     // location.reload(); // Or manually move DOM
                     // To avoid full reload, simply append child back to 'from' container
                     evt.from.appendChild(evt.item);
                }
            });
        }
    </script>
@endsection
