@extends('layouts.app')

@section('title', 'Tareas')

@section('content')
    <div class="container-fluid">
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <div class="row align-items-center">
                            <div class="col">
                                <h3 class="card-title mb-0">
                                    <i class="fas fa-tasks me-2"></i>
                                    Gestión de Tareas
                                </h3>
                                <p class="text-muted mb-0">
                                    @if (auth()->user()->hasRole('administrador'))
                                        Todas las tareas del sistema
                                    @elseif(auth()->user()->hasRole('proveedor'))
                                        Tareas de tus empresas
                                    @else
                                        Mis tareas asignadas
                                    @endif
                                </p>
                            </div>
                            <div class="col-auto">
                                <span class="badge badge-info badge-lg">
                                    Total: {{ $tareas->total() }} tareas
                                </span>
                            </div>
                        </div>
                    </div>

                    <div class="card-body">
                        <!-- Filtros -->
                        <form action="{{ route('tareas.index') }}" method="GET">
                            <div class="row mb-4">
                                <div class="col-md-3">
                                    <label class="form-label font-weight-bold text-primary"><i class="fas fa-building me-1"></i> Empresa</label>
                                    <div class="input-group">
                                        <div class="input-group-prepend">
                                            <span class="input-group-text bg-white"><i class="fas fa-city text-primary"></i></span>
                                        </div>
                                        <select name="empresa_id" class="form-control" onchange="this.form.submit()">
                                            <option value="">Todas las empresas</option>
                                            @foreach($empresas as $empresa)
                                                <option value="{{ $empresa->id }}" {{ request('empresa_id') == $empresa->id ? 'selected' : '' }}>
                                                    {{ $empresa->nombre_comercial ?? $empresa->razon_social }}
                                                </option>
                                            @endforeach
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label font-weight-bold text-success"><i class="fas fa-tasks me-1"></i> Estado</label>
                                    <div class="input-group">
                                        <div class="input-group-prepend">
                                            <span class="input-group-text bg-white"><i class="fas fa-filter text-success"></i></span>
                                        </div>
                                        <select name="estado" class="form-control" onchange="this.form.submit()">
                                            <option value="">Todos los estados</option>
                                            <option value="pendiente" class="text-warning font-weight-bold" {{ request('estado') == 'pendiente' ? 'selected' : '' }}>&#xf017; Pendiente</option>
                                            <option value="en_progreso" class="text-primary font-weight-bold" {{ request('estado') == 'en_progreso' ? 'selected' : '' }}>&#xf110; En Progreso</option>
                                            <option value="finalizado" class="text-success font-weight-bold" {{ request('estado') == 'finalizado' ? 'selected' : '' }}>&#xf058; Finalizado</option>
                                            <option value="cancelado" class="text-danger font-weight-bold" {{ request('estado') == 'cancelado' ? 'selected' : '' }}>&#xf05e; Cancelado</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label">Buscar</label>
                                    <div class="input-group">
                                        <input type="text" name="search" class="form-control"
                                            placeholder="Buscar por título..." value="{{ request('search') }}">
                                        <div class="input-group-append">
                                            <button class="btn btn-outline-secondary" type="submit">
                                                <i class="fas fa-search"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label">&nbsp;</label>
                                    <a href="{{ route('tareas.index') }}" class="btn btn-outline-secondary d-block">
                                        <i class="fas fa-times"></i> Limpiar filtros
                                    </a>
                                </div>
                            </div>
                            
                            <!-- Preserve sort params -->
                            @if(request('sort_by'))
                                <input type="hidden" name="sort_by" value="{{ request('sort_by') }}">
                                <input type="hidden" name="order" value="{{ request('order') }}">
                            @endif
                        </form>

                        <!-- Tabla de tareas -->
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead class="table-dark">
                                    <tr>
                                        <th>
                                            <a href="{{ request()->fullUrlWithQuery(['sort_by' => 'titulo', 'order' => request('order') == 'asc' ? 'desc' : 'asc']) }}" class="text-white text-decoration-none">
                                                Título
                                                @if(request('sort_by') == 'titulo')
                                                    <i class="fas fa-sort-{{ request('order') == 'asc' ? 'up' : 'down' }}"></i>
                                                @else
                                                    <i class="fas fa-sort text-muted"></i>
                                                @endif
                                            </a>
                                        </th>
                                        <th>
                                            <a href="{{ request()->fullUrlWithQuery(['sort_by' => 'empresa', 'order' => request('order') == 'asc' ? 'desc' : 'asc']) }}" class="text-white text-decoration-none">
                                                Empresa
                                                @if(request('sort_by') == 'empresa')
                                                    <i class="fas fa-sort-{{ request('order') == 'asc' ? 'up' : 'down' }}"></i>
                                                @else
                                                    <i class="fas fa-sort text-muted"></i>
                                                @endif
                                            </a>
                                        </th>
                                        @if (auth()->user()->hasRole('administrador') || auth()->user()->hasRole('proveedor'))
                                            <th>
                                                <a href="{{ request()->fullUrlWithQuery(['sort_by' => 'operador', 'order' => request('order') == 'asc' ? 'desc' : 'asc']) }}" class="text-white text-decoration-none">
                                                    Operador
                                                    @if(request('sort_by') == 'operador')
                                                        <i class="fas fa-sort-{{ request('order') == 'asc' ? 'up' : 'down' }}"></i>
                                                    @else
                                                        <i class="fas fa-sort text-muted"></i>
                                                    @endif
                                                </a>
                                            </th>
                                        @endif
                                        <th>
                                            <a href="{{ request()->fullUrlWithQuery(['sort_by' => 'estado', 'order' => request('order') == 'asc' ? 'desc' : 'asc']) }}" class="text-white text-decoration-none">
                                                Estado
                                                @if(request('sort_by') == 'estado')
                                                    <i class="fas fa-sort-{{ request('order') == 'asc' ? 'up' : 'down' }}"></i>
                                                @else
                                                    <i class="fas fa-sort text-muted"></i>
                                                @endif
                                            </a>
                                        </th>
                                        <th>Progreso</th>
                                        <th>
                                            <a href="{{ request()->fullUrlWithQuery(['sort_by' => 'fecha_inicio_estimada', 'order' => request('order') == 'asc' ? 'desc' : 'asc']) }}" class="text-white text-decoration-none">
                                                Fecha Inicio
                                                @if(request('sort_by') == 'fecha_inicio_estimada')
                                                    <i class="fas fa-sort-{{ request('order') == 'asc' ? 'up' : 'down' }}"></i>
                                                @else
                                                    <i class="fas fa-sort text-muted"></i>
                                                @endif
                                            </a>
                                        </th>
                                        <th>
                                            <a href="{{ request()->fullUrlWithQuery(['sort_by' => 'fecha_fin_estimada', 'order' => request('order') == 'asc' ? 'desc' : 'asc']) }}" class="text-white text-decoration-none">
                                                Fecha Fin
                                                @if(request('sort_by') == 'fecha_fin_estimada')
                                                    <i class="fas fa-sort-{{ request('order') == 'asc' ? 'up' : 'down' }}"></i>
                                                @else
                                                    <i class="fas fa-sort text-muted"></i>
                                                @endif
                                            </a>
                                        </th>
                                        <th>Acciones</th>
                                    </tr>
                                </thead>
                                <tbody id="tablaTareas">
                                    @forelse($tareas as $tarea)
                                        @php
                                            $rowClass = '';
                                            if ($tarea->estado != 'finalizado' && $tarea->estado != 'cancelado' && $tarea->fecha_fin_estimada) {
                                                if ($tarea->fecha_fin_estimada < now()->startOfDay()) {
                                                    $rowClass = 'table-danger';
                                                } elseif ($tarea->fecha_fin_estimada->isToday()) {
                                                    $rowClass = 'table-warning';
                                                }
                                            }
                                        @endphp
                                        <tr class="{{ $rowClass }}" 
                                            data-estado="{{ $tarea->estado }}"
                                            data-titulo="{{ strtolower($tarea->titulo) }}">
                                            <td>
                                                <div class="d-flex align-items-center">
                                                    <div class="me-3">
                                                        @switch($tarea->estado)
                                                            @case('pendiente')
                                                                <i class="fas fa-clock text-warning"></i>
                                                            @break

                                                            @case('en_progreso')
                                                                <i class="fas fa-spinner text-primary"></i>
                                                            @break

                                                            @case('finalizado')
                                                                <i class="fas fa-check-circle text-success"></i>
                                                            @break

                                                            @case('cancelado')
                                                                <i class="fas fa-ban text-danger"></i>
                                                            @break
                                                        @endswitch
                                                    </div>
                                                    <div>
                                                        <h6 class="mb-0">{{ $tarea->titulo }}</h6>
                                                        @if ($tarea->descripcion)
                                                            <small
                                                                class="text-muted">{{ Str::limit($tarea->descripcion, 50) }}</small>
                                                        @endif
                                                    </div>
                                                </div>
                                            </td>
                                            <td>
                                                <span class="badge badge-outline-info">
                                                    {{ $tarea->empresa->nombre_comercial ?? $tarea->empresa->razon_social }}
                                                </span>
                                            </td>
                                            @if (auth()->user()->hasRole('administrador') || auth()->user()->hasRole('proveedor'))
                                                <td>
                                                    <div class="d-flex align-items-center">
                                                        <div class="avatar avatar-sm me-2">
                                                            <span class="avatar-initial rounded-circle bg-secondary">
                                                                {{ substr($tarea->operador->nombre_corto, 0, 1) }}
                                                            </span>
                                                        </div>
                                                        <span>{{ $tarea->operador->nombre_corto }}</span>
                                                    </div>
                                                </td>
                                            @endif
                                            <td>
                                                @switch($tarea->estado)
                                                    @case('pendiente')
                                                        <span class="badge bg-warning">Pendiente</span>
                                                    @break

                                                    @case('en_progreso')
                                                        <span class="badge bg-primary">En Progreso</span>
                                                    @break

                                                    @case('finalizado')
                                                        <span class="badge bg-success">Finalizado</span>
                                                    @break

                                                    @case('cancelado')
                                                        <span class="badge bg-danger">Cancelado</span>
                                                    @break
                                                @endswitch
                                            </td>
                                            <td>
                                                @php
                                                    $progreso = 0;
                                                    if ($tarea->estado === 'en_progreso') {
                                                        $progreso = 50;
                                                    } elseif ($tarea->estado === 'finalizado') {
                                                        $progreso = 100;
                                                    }
                                                @endphp
                                                <div class="progress" style="height: 6px;">
                                                    <div class="progress-bar 
                                                    @if ($progreso == 0) bg-secondary
                                                    @elseif($progreso < 100) bg-primary
                                                    @else bg-success @endif"
                                                        role="progressbar" style="width: {{ $progreso }}%">
                                                    </div>
                                                </div>
                                                <small class="text-muted">{{ $progreso }}%</small>
                                            </td>
                                            <td>
                                                <div>
                                                    <small class="text-muted">Programada:</small><br>
                                                    {{ $tarea->fecha_inicio_estimada ? $tarea->fecha_inicio_estimada->format('d/m/Y') : 'N/A' }}
                                                </div>
                                                @if ($tarea->fecha_inicio_real)
                                                    <div class="mt-1">
                                                        <small class="text-success">Real:</small><br>
                                                        <small
                                                            class="text-success">{{ $tarea->fecha_inicio_real->format('d/m/Y') }}</small>
                                                    </div>
                                                @endif
                                            </td>
                                            <td>
                                                <div>
                                                    <small class="text-muted">Programada:</small><br>
                                                    {{ $tarea->fecha_fin_estimada ? $tarea->fecha_fin_estimada->format('d/m/Y') : 'N/A' }}
                                                </div>
                                                @if ($tarea->fecha_fin_real)
                                                    <div class="mt-1">
                                                        <small class="text-success">Real:</small><br>
                                                        <small
                                                            class="text-success">{{ $tarea->fecha_fin_real->format('d/m/Y') }}</small>
                                                    </div>
                                                @endif
                                            </td>
                                            <td>
                                                <div class="btn-group" role="group">
                                                    <button type="button" 
                                                        class="btn btn-sm btn-outline-primary" 
                                                        title="Ver detalles"
                                                        onclick="showTareaDetails({{ $tarea->id }})">
                                                        <i class="fas fa-eye"></i>
                                                    </button>

                                                    @can('update', $tarea)
                                                        <a href="{{ route('tareas.edit', $tarea) }}"
                                                            class="btn btn-sm btn-outline-warning" title="Editar tarea">
                                                            <i class="fas fa-edit"></i>
                                                        </a>
                                                    @endcan

                                                    @can('updateStatus', $tarea)
                                                        @if($tarea->estado === 'finalizado')
                                                            <button type="button" class="btn btn-sm btn-outline-success" 
                                                                onclick="verSolucion('{{ addslashes($tarea->observaciones_finales) }}')"
                                                                title="Ver Solución">
                                                                <i class="fas fa-check-double"></i>
                                                            </button>
                                                        @else
                                                            <div class="btn-group" role="group">
                                                                <button type="button"
                                                                    class="btn btn-sm btn-outline-info dropdown-toggle"
                                                                    data-toggle="dropdown" title="Cambiar estado">
                                                                    <i class="fas fa-exchange-alt"></i>
                                                                </button>
                                                                <ul class="dropdown-menu">
                                                                    @if ($tarea->estado != 'pendiente')
                                                                        <li>
                                                                            <a class="dropdown-item" href="#"
                                                                                onclick="cambiarEstado({{ $tarea->id }}, 'pendiente')">
                                                                                <i class="fas fa-clock text-warning me-2"></i>
                                                                                Pendiente
                                                                            </a>
                                                                        </li>
                                                                    @endif
                                                                    @if ($tarea->estado != 'en_progreso')
                                                                        <li>
                                                                            <a class="dropdown-item" href="#"
                                                                                onclick="cambiarEstado({{ $tarea->id }}, 'en_progreso')">
                                                                                <i class="fas fa-spinner text-primary me-2"></i>
                                                                                En Progreso
                                                                            </a>
                                                                        </li>
                                                                    @endif
                                                                    @if ($tarea->estado != 'finalizado')
                                                                        <li>
                                                                            <a class="dropdown-item" href="#"
                                                                                onclick="cambiarEstado({{ $tarea->id }}, 'finalizado')">
                                                                                <i class="fas fa-check-circle text-success me-2"></i>
                                                                                Finalizado
                                                                            </a>
                                                                        </li>
                                                                    @endif
                                                                    @if ($tarea->estado != 'cancelado')
                                                                        <li>
                                                                            <a class="dropdown-item" href="#"
                                                                                onclick="cambiarEstado({{ $tarea->id }}, 'cancelado')">
                                                                                <i class="fas fa-ban text-danger me-2"></i>
                                                                                Cancelado
                                                                            </a>
                                                                        </li>
                                                                    @endif
                                                                </ul>
                                                            </div>
                                                        @endif
                                                    @endcan

                                                    @can('delete', $tarea)
                                                        <form action="{{ route('tareas.destroy', $tarea) }}" method="POST" class="d-inline" onsubmit="return confirm('¿Estás seguro de eliminar esta tarea?');">
                                                            @csrf
                                                            @method('DELETE')
                                                            <button type="submit" class="btn btn-sm btn-outline-danger" title="Eliminar tarea">
                                                                <i class="fas fa-trash"></i>
                                                            </button>
                                                        </form>
                                                    @endcan
                                                </div>
                                            </td>
                                        </tr>
                                        @empty
                                            <tr>
                                                <td colspan="8" class="text-center py-4">
                                                    <div class="empty-state">
                                                        <i class="fas fa-tasks text-muted mb-3"
                                                            style="font-size: 3rem;"></i>
                                                        <h4 class="text-muted">No hay tareas</h4>
                                                        <p class="text-muted mb-0">
                                                            @if (auth()->user()->hasRole('administrador'))
                                                                No hay tareas creadas en el sistema
                                                            @elseif(auth()->user()->hasRole('proveedor'))
                                                                No tienes tareas en tus empresas
                                                            @else
                                                                No tienes tareas asignadas
                                                            @endif
                                                        </p>
                                                    </div>
                                                </td>
                                            </tr>
                                        @endforelse
                                    </tbody>
                                </table>
                            </div>

                            <!-- Paginación -->
                            @if ($tareas->hasPages())
                                <div class="row align-items-center">
                                    <div class="col-sm-12 col-md-5">
                                        <div class="text-muted">
                                            Mostrando {{ $tareas->firstItem() }} a {{ $tareas->lastItem() }}
                                            de {{ $tareas->total() }} tareas
                                        </div>
                                    </div>
                                    <div class="col-sm-12 col-md-7">
                                        {{ $tareas->links() }}
                                    </div>
                                </div>
                            @endif
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Modal para cambiar estado -->
        <div class="modal fade" id="cambiarEstadoModal" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <form id="cambiarEstadoForm" method="POST" enctype="multipart/form-data">
                        @csrf
                        @method('PATCH')
                        <div class="modal-header">
                            <h5 class="modal-title">Cambiar Estado de la Tarea</h5>
                            <button type="button" class="close" data-dismiss="modal">
                                <span>&times;</span>
                            </button>
                        </div>
                        <div class="modal-body">
                            <p>¿Estás seguro de que deseas cambiar el estado de esta tarea?</p>
                            <input type="hidden" name="estado" id="nuevoEstado">
                            
                            <div class="form-group" id="observacionesContainer" style="display: none;">
                                <label for="observaciones_finales" class="font-weight-bold">Observaciones Finales (Solución):</label>
                                <textarea class="form-control text-uppercase" name="observaciones_finales" id="observacionesFinales" rows="3" 
                                    placeholder="Describa los procesos o cambios realizados..."
                                    oninput="this.value = this.value.toUpperCase()"></textarea>
                                
                                <div class="mt-3">
                                    <label for="evidencia" class="font-weight-bold">Adjuntar Evidencia (Opcional):</label>
                                    <input type="file" class="form-control-file" name="evidencia" id="inputEvidencia">
                                    <small class="text-muted d-block">Formatos permitidos: Imágenes, PDF, Docs, Excel, Archivos comprimidos (Max 10MB).</small>
                                </div>

                                <small class="text-muted mt-2 d-block">Esta información no podrá ser modificada una vez guardada.</small>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancelar</button>
                            <button type="submit" class="btn btn-primary">Confirmar</button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
        <!-- Modal para ver solución -->
        <div class="modal fade" id="verSolucionModal" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header bg-success text-white">
                        <h5 class="modal-title">
                            <i class="fas fa-check-double mr-2"></i> Solución de la Tarea
                        </h5>
                        <button type="button" class="close text-white" data-dismiss="modal">
                            <span>&times;</span>
                        </button>
                    </div>
                    <div class="modal-body">
                        <div class="alert alert-light border">
                            <p id="textoSolucion" class="mb-0" style="white-space: pre-wrap;"></p>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">Cerrar</button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Modal para ver detalles de tarea -->
        <div class="modal fade" id="showTareaModal" tabindex="-1" role="dialog" aria-labelledby="showTareaModalLabel" aria-hidden="true">
            <div class="modal-dialog modal-lg" role="document">
                <div class="modal-content" id="showTareaModalContent">
                     <div class="modal-header">
                        <h5 class="modal-title">Cargando...</h5>
                        <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                            <span aria-hidden="true">&times;</span>
                        </button>
                    </div>
                    <div class="modal-body text-center py-5">
                        <div class="spinner-border text-primary" role="status">
                            <span class="sr-only">Cargando...</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
@endsection

@section('scripts')
    <script>
        let currentTareaId = null;

        function showTareaDetails(id) {
            currentTareaId = id;
            $('#showTareaModal').modal('show');
            
            // Show loading state
            $('#showTareaModalContent').html(`
                <div class="modal-header">
                    <h5 class="modal-title">Cargando...</h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">&times;</span>
                    </button>
                </div>
                <div class="modal-body text-center py-5">
                    <div class="spinner-border text-primary" role="status">
                        <span class="sr-only">Cargando...</span>
                    </div>
                </div>
            `);

            loadTareaDetails(id);
        }

        function loadTareaDetails(id) {
            var url = '{{ route("tareas.show", ":id") }}'.replace(':id', id);
            $.get(url, function(data) {
                $('#showTareaModalContent').html(data);
            }).fail(function() {
                $('#showTareaModalContent').html('<div class="modal-body text-danger">Error al cargar la tarea.</div>');
            });
        }

        // Handle Comment Form Submit using Event Delegation
        $(document).on('submit', '#addCommentForm', function(e) {
            e.preventDefault();
            var form = $(this);
            var url = form.attr('action');
            var formData = form.serialize();
            var btn = form.find('button[type="submit"]');
            
            btn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Enviando...');

            $.post(url, formData, function(response) {
                if(response.success || response.id) { // Check for success or created object
                    loadTareaDetails(currentTareaId); // Reload modal to show new comment
                } else {
                    alert('Error al enviar el comentario.');
                    btn.prop('disabled', false).html('<i class="fas fa-paper-plane"></i> Enviar');
                }
            }).fail(function() {
                alert('Error de conexión.');
                btn.prop('disabled', false).html('<i class="fas fa-paper-plane"></i> Enviar');
            });
        });

        function verSolucion(solucion) {
            document.getElementById('textoSolucion').innerText = solucion || 'Sin observaciones registradas.';
            $('#verSolucionModal').modal('show');
        }

        function cambiarEstado(tareaId, nuevoEstado) {
            const form = document.getElementById('cambiarEstadoForm');
            // Usar la función route() de Laravel para generar la ruta correcta incluyendo subcarpetas y esquema
            var url = '{{ route("tareas.update-status", ":id") }}';
            form.action = url.replace(':id', tareaId);
            document.getElementById('nuevoEstado').value = nuevoEstado;
            
            // Mostrar campo de observaciones si es finalizado
            const obsContainer = document.getElementById('observacionesContainer');
            const obsInput = document.getElementById('observacionesFinales');
            const fileInput = document.getElementById('inputEvidencia');
            
            if (nuevoEstado === 'finalizado') {
                obsContainer.style.display = 'block';
                obsInput.required = true;
                fileInput.required = false; // Opcional
            } else {
                obsContainer.style.display = 'none';
                obsInput.required = false;
                obsInput.value = ''; // Limpiar si no es finalizado
                fileInput.value = ''; // Limpiar archivo
                fileInput.required = false;
            }
            
            $('#cambiarEstadoModal').modal('show');
        }

        function limpiarFiltros() {
            window.location.href = '{{ route('tareas.index') }}';
        }

        // AJAX submit for Status Change
        $(document).on('submit', '#cambiarEstadoForm', function(e) {
            e.preventDefault();
            var form = $(this);
            var url = form.attr('action');
            var formData = new FormData(this);
            var btn = form.find('button[type="submit"]');
            var cancelBtn = form.find('button[type="button"]'); // Cancel button
            var originalText = btn.html();

            // Disable buttons and show loading
            btn.prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Guardando...');
            cancelBtn.prop('disabled', true);

            $.ajax({
                url: url,
                type: 'POST',
                data: formData,
                processData: false,
                contentType: false,
                headers: {
                    'Accept': 'application/json',
                    'X-Requested-With': 'XMLHttpRequest'
                },
                success: function(response) {
                    $('#cambiarEstadoModal').modal('hide');
                    // Reload page to show changes. 
                    // location.reload() preserves query parameters (filters)
                    location.reload();
                },
                error: function(xhr) {
                    btn.prop('disabled', false).html(originalText);
                    cancelBtn.prop('disabled', false);
                    alert('Error al actualizar el estado. Por favor intente nuevamente.');
                    console.error(xhr.responseText);
                }
            });
        });
    </script>
@endsection