<?php

namespace App\Policies;

use App\Models\Tarea;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class TareaPolicy
{
    use HandlesAuthorization;

    public function viewAny(User $user)
    {
        return $user->hasAnyRole(['administrador', 'proveedor', 'operador']);
    }

    public function view(User $user, Tarea $tarea)
    {
        // Administrador puede ver todos
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede ver tareas de sus empresas
        if ($user->hasRole('proveedor')) {
            return $tarea->empresa->proveedores->contains($user->id);
        }

        // Operador puede ver solo tareas asignadas a él
        if ($user->hasRole('operador')) {
            return $tarea->operador_id == $user->id;
        }

        return false;
    }

    public function create(User $user)
    {
        // Solo administradores y proveedores (dependiendo de la lógica)
        return $user->hasRole('administrador') || $user->hasRole('proveedor');
    }

    public function update(User $user, Tarea $tarea)
    {
        // Administrador puede actualizar todos
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede actualizar tareas de sus empresas
        if ($user->hasRole('proveedor')) {
            return $tarea->empresa->proveedores->contains($user->id);
        }

        return false;
    }

    public function delete(User $user, Tarea $tarea)
    {
        // Administrador puede eliminar todos
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede eliminar tareas de sus empresas solo si están pendientes o canceladas
        if ($user->hasRole('proveedor')) {
            return $tarea->empresa->proveedores->contains($user->id) && in_array($tarea->estado, ['pendiente', 'cancelado']);
        }

        return false;
    }

    public function updateStatus(User $user, Tarea $tarea)
    {
        // Administrador puede cambiar cualquier estado
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Operador puede cambiar estado solo de sus tareas asignadas
        if ($user->hasRole('operador')) {
            return $tarea->operador_id == $user->id;
        }

        return false;
    }

    public function uploadEvidence(User $user, Tarea $tarea)
    {
        // Administrador puede subir evidencias a cualquier tarea
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Operador puede subir evidencias solo a sus tareas asignadas
        if ($user->hasRole('operador')) {
            return $tarea->operador_id == $user->id;
        }

        return false;
    }

    public function uploadMaterial(User $user, Tarea $tarea)
    {
        // Administrador puede subir material a cualquier tarea
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede subir material a tareas de sus empresas
        if ($user->hasRole('proveedor')) {
            return $tarea->empresa->proveedores->contains($user->id);
        }

        return false;
    }

    public function comment(User $user, Tarea $tarea)
    {
        // Administrador puede comentar en cualquier tarea
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede comentar en tareas de sus empresas
        if ($user->hasRole('proveedor')) {
            return $tarea->empresa->proveedores->contains($user->id);
        }

        // Operador puede comentar en sus tareas asignadas
        if ($user->hasRole('operador')) {
            return $tarea->operador_id == $user->id;
        }

        return false;
    }
}
