<?php
// app/Policies/SerieProyectoPolicy.php

namespace App\Policies;

use App\Models\SerieProyecto;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class SerieProyectoPolicy
{
    use HandlesAuthorization;

    public function viewAny(User $user)
    {
        return $user->hasAnyRole(['administrador', 'proveedor', 'operador']);
    }

    public function view(User $user, SerieProyecto $serie)
    {
        // Administrador puede ver todas las series
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede ver solo sus propias series
        if ($user->hasRole('proveedor')) {
            return $serie->proveedor_id === $user->id;
        }

        // Operador puede ver series que tengan proyectos asignados a él
        if ($user->hasRole('operador')) {
            return $serie->proyectos()->where('operador_id', $user->id)->exists();
        }

        return false;
    }

    public function create(User $user)
    {
        return $user->hasRole('proveedor');
    }

    public function update(User $user, SerieProyecto $serie)
    {
        if ($user->hasRole('administrador')) {
            return true;
        }

        if ($user->hasRole('proveedor')) {
            return $serie->proveedor_id === $user->id && $serie->estado === 'pendiente';
        }

        return false;
    }

    public function delete(User $user, SerieProyecto $serie)
    {
        if ($user->hasRole('administrador')) {
            return true;
        }

        if ($user->hasRole('proveedor')) {
            return $serie->proveedor_id === $user->id && $serie->estado === 'pendiente';
        }

        return false;
    }

    public function approve(User $user, SerieProyecto $serie)
    {
        return $user->hasRole('administrador') && $serie->estado === 'pendiente';
    }

    public function reject(User $user, SerieProyecto $serie)
    {
        return $user->hasRole('administrador') && $serie->estado === 'pendiente';
    }

    public function addProjects(User $user, SerieProyecto $serie)
    {
        // Administrador puede agregar proyectos a cualquier serie aprobada
        if ($user->hasRole('administrador')) {
            return $serie->estado === 'aprobada';
        }

        // Proveedor puede agregar proyectos solo a sus series aprobadas
        if ($user->hasRole('proveedor')) {
            return $serie->proveedor_id === $user->id && $serie->estado === 'aprobada';
        }

        return false;
    }
}
