<?php
// app/Policies/ProyectoPolicy.php

namespace App\Policies;

use App\Models\Proyecto;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class ProyectoPolicy
{
    use HandlesAuthorization;

    public function viewAny(User $user)
    {
        return $user->hasAnyRole(['administrador', 'proveedor', 'operador']);
    }

    public function view(User $user, Proyecto $proyecto)
    {
        // Administrador puede ver todos
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede ver proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // Operador puede ver solo proyectos asignados a él
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }

    public function create(User $user)
    {
        // Solo proveedores pueden crear proyectos (dentro de series)
        return $user->hasRole('proveedor');
    }

    public function update(User $user, Proyecto $proyecto)
    {
        // Administrador puede actualizar todos
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede actualizar proyectos de sus series
        if ($user->hasRole('proveedor')) {
            //  return true;
            return $proyecto->serie->proveedor_id === $user->id;
        }

        return false;
    }

    public function delete(User $user, Proyecto $proyecto)
    {
        // Administrador puede eliminar todos
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede eliminar proyectos de sus series solo si están pendientes
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id && $proyecto->estado === 'pendiente';
        }

        return false;
    }

    public function updateStatus(User $user, Proyecto $proyecto)
    {
        // Administrador puede cambiar cualquier estado
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Operador puede cambiar estado solo de sus proyectos asignados
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }

    public function uploadEvidence(User $user, Proyecto $proyecto)
    {
        // Administrador puede subir evidencias a cualquier proyecto
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Operador puede subir evidencias solo a sus proyectos
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }

    public function comment(User $user, Proyecto $proyecto)
    {
        // Administrador puede comentar en cualquier proyecto
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede comentar en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // Operador puede comentar en sus proyectos asignados
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }
}
