<?php
// app/Policies/NotificacionPolicy.php

namespace App\Policies;

use App\Models\Notificacion;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class NotificacionPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user)
    {
        // Todos los usuarios autenticados pueden ver sus notificaciones
        return true;
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Notificacion $notificacion)
    {
        // Solo puede ver sus propias notificaciones
        return $notificacion->user_id === $user->id;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user)
    {
        // Solo administradores pueden crear notificaciones manualmente
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Notificacion $notificacion)
    {
        // Solo puede actualizar sus propias notificaciones
        return $notificacion->user_id === $user->id;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Notificacion $notificacion)
    {
        // Administrador puede eliminar cualquier notificación
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Usuario puede eliminar solo sus propias notificaciones
        return $notificacion->user_id === $user->id;
    }

    /**
     * Determine whether the user can mark notification as read.
     */
    public function markAsRead(User $user, Notificacion $notificacion)
    {
        // Solo puede marcar como leída sus propias notificaciones
        return $notificacion->user_id === $user->id;
    }

    /**
     * Determine whether the user can mark all notifications as read.
     */
    public function markAllAsRead(User $user)
    {
        // Cualquier usuario puede marcar todas sus notificaciones como leídas
        return true;
    }

    /**
     * Determine whether the user can send notifications to others.
     */
    public function sendToOthers(User $user)
    {
        // Solo administradores pueden enviar notificaciones a otros usuarios
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can send broadcast notifications.
     */
    public function broadcast(User $user)
    {
        // Solo administradores pueden enviar notificaciones masivas
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can view notification statistics.
     */
    public function viewStatistics(User $user)
    {
        // Solo administradores pueden ver estadísticas de notificaciones
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can export notifications.
     */
    public function export(User $user)
    {
        // Solo administradores pueden exportar notificaciones
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can clear all notifications.
     */
    public function clearAll(User $user)
    {
        // Cualquier usuario puede limpiar todas sus notificaciones
        return true;
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Notificacion $notificacion)
    {
        // Solo administradores pueden restaurar notificaciones eliminadas
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Notificacion $notificacion)
    {
        // Solo administradores pueden eliminar permanentemente
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can configure notification preferences.
     */
    public function configurePreferences(User $user)
    {
        // Cualquier usuario puede configurar sus preferencias de notificación
        return true;
    }

    /**
     * Determine whether the user can view unread count.
     */
    public function viewUnreadCount(User $user)
    {
        // Cualquier usuario puede ver el conteo de sus notificaciones no leídas
        return true;
    }
}
