<?php
// app/Policies/EvidenciaPolicy.php

namespace App\Policies;

use App\Models\Evidencia;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class EvidenciaPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user)
    {
        // Todos los usuarios autenticados pueden ver evidencias según sus permisos
        return $user->hasAnyRole(['administrador', 'proveedor', 'operador']);
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Evidencia $evidencia)
    {
        // Administrador puede ver todas las evidencias
        if ($user->hasRole('administrador')) {
            return true;
        }

        $tarea = $evidencia->tarea;

        // Proveedor puede ver evidencias de tareas de sus empresas
        if ($user->hasRole('proveedor')) {
            return $tarea->empresa->proveedor_id == $user->id;
        }

        // Operador puede ver evidencias de sus tareas asignadas
        if ($user->hasRole('operador')) {
            return $tarea->operador_id == $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user)
    {
        // Solo administradores y operadores pueden subir evidencias
        return $user->hasAnyRole(['administrador', 'operador']);
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Evidencia $evidencia)
    {
        // Administrador puede actualizar cualquier evidencia
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Solo el usuario que subió la evidencia puede actualizarla
        return $evidencia->user_id === $user->id;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Evidencia $evidencia)
    {
        // Administrador puede eliminar cualquier evidencia
        if ($user->hasRole('administrador')) {
            return true;
        }

        // El usuario que subió la evidencia puede eliminarla
        if ($evidencia->user_id == $user->id) {
            return true;
        }

        // Proveedor puede eliminar evidencias de tareas de sus empresas
        if ($user->hasRole('proveedor')) {
            return $evidencia->tarea->empresa->proveedor_id == $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can download the evidence.
     */
    public function download(User $user, Evidencia $evidencia)
    {
        // Administrador puede descargar cualquier evidencia
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $evidencia->proyecto;

        // Proveedor puede descargar evidencias de proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // Operador puede descargar evidencias de sus proyectos asignados
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can upload evidence to a specific project.
     */
    public function uploadToProject(User $user, $proyectoId)
    {
        // Administrador puede subir evidencias a cualquier proyecto
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Operador puede subir evidencias solo a sus proyectos asignados
        if ($user->hasRole('operador')) {
            return \App\Models\Proyecto::where('id', $proyectoId)
                ->where('operador_id', $user->id)
                ->exists();
        }

        return false;
    }

    /**
     * Determine whether the user can replace the evidence file.
     */
    public function replace(User $user, Evidencia $evidencia)
    {
        // Administrador puede reemplazar cualquier evidencia
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Solo el usuario que subió la evidencia puede reemplazarla
        return $evidencia->user_id === $user->id;
    }

    /**
     * Determine whether the user can approve the evidence.
     */
    public function approve(User $user, Evidencia $evidencia)
    {
        // Solo administradores pueden aprobar evidencias
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $evidencia->proyecto;

        // Proveedor puede aprobar evidencias de proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can reject the evidence.
     */
    public function reject(User $user, Evidencia $evidencia)
    {
        // Solo administradores pueden rechazar evidencias
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $evidencia->proyecto;

        // Proveedor puede rechazar evidencias de proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can view evidence metadata.
     */
    public function viewMetadata(User $user, Evidencia $evidencia)
    {
        // Administrador puede ver metadatos de cualquier evidencia
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $evidencia->proyecto;

        // Proveedor puede ver metadatos de evidencias de sus proyectos
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // Operador puede ver metadatos de evidencias de sus proyectos
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can share the evidence.
     */
    public function share(User $user, Evidencia $evidencia)
    {
        // Administrador puede compartir cualquier evidencia
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $evidencia->proyecto;

        // Proveedor puede compartir evidencias de proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // El usuario que subió la evidencia puede compartirla
        return $evidencia->user_id === $user->id;
    }

    /**
     * Determine whether the user can export evidence data.
     */
    public function export(User $user, Evidencia $evidencia)
    {
        // Solo administradores pueden exportar datos de evidencias
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $evidencia->proyecto;

        // Proveedor puede exportar evidencias de proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Evidencia $evidencia)
    {
        // Solo administradores pueden restaurar evidencias eliminadas
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Evidencia $evidencia)
    {
        // Solo administradores pueden eliminar permanentemente
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can view evidence history.
     */
    public function viewHistory(User $user, Evidencia $evidencia)
    {
        // Administrador puede ver historial de cualquier evidencia
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $evidencia->proyecto;

        // Proveedor puede ver historial de evidencias de sus proyectos
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // El usuario que subió la evidencia puede ver su historial
        return $evidencia->user_id === $user->id;
    }

    /**
     * Determine whether the user can comment on the evidence.
     */
    public function comment(User $user, Evidencia $evidencia)
    {
        // Administrador puede comentar en cualquier evidencia
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $evidencia->proyecto;

        // Proveedor puede comentar en evidencias de proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id == $user->id;
        }

        // Operador puede comentar en evidencias de sus proyectos asignados
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id == $user->id;
        }

        return false;
    }
}
