<?php

namespace App\Policies;

use App\Models\Empresa;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class EmpresaPolicy
{
    use HandlesAuthorization;

    public function viewAny(User $user)
    {
        return $user->hasAnyRole(['administrador', 'proveedor', 'operador']);
    }

    public function view(User $user, Empresa $empresa)
    {
        if ($user->hasRole('administrador')) {
            return true;
        }

        if ($user->hasRole('proveedor')) {
            return $empresa->proveedores->contains($user->id);
        }

        if ($user->hasRole('operador')) {
            return $empresa->tareas()->where('operador_id', $user->id)->exists();
        }

        return false;
    }

    public function create(User $user)
    {
        return $user->hasRole('proveedor') || $user->hasRole('administrador');
    }

    public function update(User $user, Empresa $empresa)
    {
        if ($user->hasRole('administrador')) {
            return true;
        }

        if ($user->hasRole('proveedor')) {
            // Permitir editar si es uno de los proveedores asignados
            return $empresa->proveedores->contains($user->id);
        }

        return false;
    }

    public function addProjects(User $user, Empresa $empresa)
    {
        if ($user->hasRole('administrador')) {
            return true;
        }

        if ($user->hasRole('proveedor')) {
            return $empresa->proveedores->contains($user->id);
        }

        return false;
    }

    public function approve(User $user, Empresa $empresa)
    {
        return $user->hasRole('administrador') && $empresa->estado === 'pendiente';
    }

    public function reject(User $user, Empresa $empresa)
    {
        return $user->hasRole('administrador') && $empresa->estado === 'pendiente';
    }

    public function delete(User $user, Empresa $empresa)
    {
        if ($user->hasRole('administrador')) {
            return true;
        }

        if ($user->hasRole('proveedor')) {
            return $empresa->proveedores->contains($user->id);
        }

        return false;
    }
}
