<?php
// app/Policies/ComentarioPolicy.php

namespace App\Policies;

use App\Models\Comentario;
use App\Models\User;
use Illuminate\Auth\Access\HandlesAuthorization;

class ComentarioPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any models.
     */
    public function viewAny(User $user)
    {
        // Todos los usuarios autenticados pueden ver comentarios según sus permisos
        return $user->hasAnyRole(['administrador', 'proveedor', 'operador']);
    }

    /**
     * Determine whether the user can view the model.
     */
    public function view(User $user, Comentario $comentario)
    {
        // Administrador puede ver todos los comentarios
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $comentario->proyecto;

        // Proveedor puede ver comentarios en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id == $user->id;
        }

        // Operador puede ver comentarios en sus proyectos asignados
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id == $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can create models.
     */
    public function create(User $user)
    {
        // Todos los usuarios autenticados pueden crear comentarios
        return $user->hasAnyRole(['administrador', 'proveedor', 'operador']);
    }

    /**
     * Determine whether the user can comment on a specific project.
     */
    public function commentOnProject(User $user, $proyectoId)
    {
        // Administrador puede comentar en cualquier proyecto
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = \App\Models\Proyecto::find($proyectoId);

        if (!$proyecto) {
            return false;
        }

        // Proveedor puede comentar en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // Operador puede comentar en sus proyectos asignados
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can update the model.
     */
    public function update(User $user, Comentario $comentario)
    {
        // Administrador puede actualizar cualquier comentario
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Solo el autor del comentario puede actualizarlo
        // Y solo dentro de los primeros 30 minutos
        if ($comentario->user_id == $user->id) {
            return $comentario->created_at->diffInMinutes(now()) <= 30;
        }

        return false;
    }

    /**
     * Determine whether the user can delete the model.
     */
    public function delete(User $user, Comentario $comentario)
    {
        // Administrador puede eliminar cualquier comentario
        if ($user->hasRole('administrador')) {
            return true;
        }

        // El autor del comentario puede eliminarlo
        if ($comentario->user_id == $user->id) {
            return true;
        }

        // Proveedor puede eliminar comentarios en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $comentario->proyecto->serie->proveedor_id == $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can reply to the comment.
     */
    public function reply(User $user, Comentario $comentario)
    {
        // Administrador puede responder a cualquier comentario
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $comentario->proyecto;

        // Proveedor puede responder a comentarios en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // Operador puede responder a comentarios en sus proyectos asignados
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can like/unlike the comment.
     */
    public function like(User $user, Comentario $comentario)
    {
        // Administrador puede dar like a cualquier comentario
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $comentario->proyecto;

        // Proveedor puede dar like a comentarios en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // Operador puede dar like a comentarios en sus proyectos asignados
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can mark comment as important.
     */
    public function markAsImportant(User $user, Comentario $comentario)
    {
        // Solo administradores pueden marcar comentarios como importantes
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $comentario->proyecto;

        // Proveedor puede marcar comentarios como importantes en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can pin the comment.
     */
    public function pin(User $user, Comentario $comentario)
    {
        // Solo administradores pueden fijar comentarios
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $comentario->proyecto;

        // Proveedor puede fijar comentarios en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can resolve the comment.
     */
    public function resolve(User $user, Comentario $comentario)
    {
        // Administrador puede resolver cualquier comentario
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $comentario->proyecto;

        // Proveedor puede resolver comentarios en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // El autor del comentario puede resolverlo
        if ($comentario->user_id === $user->id) {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can view comment history.
     */
    public function viewHistory(User $user, Comentario $comentario)
    {
        // Administrador puede ver historial de cualquier comentario
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $comentario->proyecto;

        // Proveedor puede ver historial de comentarios en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // El autor del comentario puede ver su historial
        if ($comentario->user_id === $user->id) {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can export comments.
     */
    public function export(User $user)
    {
        // Solo administradores pueden exportar comentarios
        if ($user->hasRole('administrador')) {
            return true;
        }

        // Proveedor puede exportar comentarios de sus proyectos
        if ($user->hasRole('proveedor')) {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can moderate comments.
     */
    public function moderate(User $user, Comentario $comentario)
    {
        // Solo administradores pueden moderar comentarios
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $comentario->proyecto;

        // Proveedor puede moderar comentarios en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can restore the model.
     */
    public function restore(User $user, Comentario $comentario)
    {
        // Solo administradores pueden restaurar comentarios eliminados
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can permanently delete the model.
     */
    public function forceDelete(User $user, Comentario $comentario)
    {
        // Solo administradores pueden eliminar permanentemente
        return $user->hasRole('administrador');
    }

    /**
     * Determine whether the user can mention others in comments.
     */
    public function mention(User $user, Comentario $comentario)
    {
        // Administrador puede mencionar a cualquiera
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $comentario->proyecto;

        // Proveedor puede mencionar en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // Operador puede mencionar en sus proyectos asignados
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }

    /**
     * Determine whether the user can attach files to comments.
     */
    public function attachFile(User $user, Comentario $comentario)
    {
        // Administrador puede adjuntar archivos a cualquier comentario
        if ($user->hasRole('administrador')) {
            return true;
        }

        $proyecto = $comentario->proyecto;

        // Proveedor puede adjuntar archivos en proyectos de sus series
        if ($user->hasRole('proveedor')) {
            return $proyecto->serie->proveedor_id === $user->id;
        }

        // Operador puede adjuntar archivos en sus proyectos asignados
        if ($user->hasRole('operador')) {
            return $proyecto->operador_id === $user->id;
        }

        return false;
    }
}
