<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Tarea extends Model
{
    use HasFactory;
    
    protected $table = 'tareas';

    protected $fillable = [
        'titulo',
        'descripcion',
        'estado',
        'prioridad',
        'fecha_inicio_estimada',
        'fecha_fin_estimada',
        'fecha_inicio_real',
        'fecha_fin_real',
        'empresa_id',
        'operador_id',
        'creador_id',
        'observaciones_finales'
    ];

    protected $casts = [
        'fecha_inicio_estimada' => 'date',
        'fecha_fin_estimada' => 'date',
        'fecha_inicio_real' => 'date',
        'fecha_fin_real' => 'date',
    ];

    // Mutadores para guardar en mayúsculas
    public function setTituloAttribute($value)
    {
        $this->attributes['titulo'] = mb_strtoupper($value, 'UTF-8');
    }

    public function setDescripcionAttribute($value)
    {
        $this->attributes['descripcion'] = $value ? mb_strtoupper($value, 'UTF-8') : null;
    }

    public function setObservacionesFinalesAttribute($value)
    {
        $this->attributes['observaciones_finales'] = $value ? mb_strtoupper($value, 'UTF-8') : null;
    }

    // Relaciones
    public function empresa()
    {
        return $this->belongsTo(Empresa::class, 'empresa_id');
    }

    public function operador()
    {
        return $this->belongsTo(User::class, 'operador_id');
    }

    public function comentarios()
    {
        return $this->hasMany(Comentario::class);
    }

    public function evidencias()
    {
        return $this->hasMany(Evidencia::class);
    }

    public function materiales()
    {
        return $this->hasMany(Material::class);
    }

    // Scopes
    public function scopePorOperador($query, $operadorId)
    {
        return $query->where('operador_id', $operadorId);
    }

    public function scopeEnProgreso($query)
    {
        return $query->where('estado', 'en_progreso');
    }

    public function scopePendientes($query)
    {
        return $query->where('estado', 'pendiente');
    }

    // Métodos auxiliares
    public function getDiasRestantesAttribute()
    {
        if ($this->estado === 'finalizado') return 0;

        $fechaLimite = $this->fecha_fin_estimada;
        $hoy = now()->startOfDay();

        return $fechaLimite->diffInDays($hoy, false);
    }

    public function getEstadoColorAttribute()
    {
        $colores = [
            'pendiente' => 'warning',
            'en_progreso' => 'primary',
            'finalizado' => 'success',
        ];

        return $colores[$this->estado] ?? 'secondary';
    }

    public function getPrioridadColorAttribute()
    {
        $colores = [
            'baja' => 'secondary',
            'media' => 'info',
            'alta' => 'warning',
            'urgente' => 'danger',
        ];

        return $colores[$this->prioridad] ?? 'secondary';
    }
}
