<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Empresa extends Model
{
    use HasFactory;

    protected $table = 'empresas';

    protected $fillable = [
        'ruc',
        'razon_social',
        'nombre_comercial',
        'usuario_sol',
        'clave_sol',
        'serie',
        'sitio_web',
        'usuario_secundario_dni',
        'usuario_secundario_user',
        'usuario_secundario_clave',
        'clave_certificado',
        'descripcion',
        'fecha_solicitud',
        'estado',
        'proveedor_id',
        'aprobado_por',
        'fecha_aprobacion',
        'comentarios_aprobacion',
        'modo_pago',
    ];

    protected $casts = [
        'fecha_solicitud' => 'date',
        'fecha_aprobacion' => 'datetime',
    ];

    // Mutadores para guardar en mayúsculas
    public function setRazonSocialAttribute($value)
    {
        $this->attributes['razon_social'] = mb_strtoupper($value, 'UTF-8');
    }

    public function setNombreComercialAttribute($value)
    {
        $this->attributes['nombre_comercial'] = $value ? mb_strtoupper($value, 'UTF-8') : null;
    }

    public function setDescripcionAttribute($value)
    {
        $this->attributes['descripcion'] = $value ? mb_strtoupper($value, 'UTF-8') : null;
    }

    // Relaciones
    public function proveedores()
    {
        return $this->belongsToMany(User::class, 'empresa_user', 'empresa_id', 'user_id');
    }

    public function proveedor()
    {
        // Mantenemos esto temporalmente por retrocompatibilidad, pero debería usarse proveedores()
        return $this->belongsTo(User::class, 'proveedor_id');
    }

    public function aprobadoPor()
    {
        return $this->belongsTo(User::class, 'aprobado_por');
    }

    public function tareas()
    {
        return $this->hasMany(Tarea::class, 'empresa_id')->orderBy('orden');
    }

    public function otrosRucs()
    {
        return $this->hasMany(EmpresaRuc::class, 'empresa_id');
    }

    public function anydeskTerminals()
    {
        return $this->hasMany(AnydeskTerminal::class, 'empresa_id');
    }

    // Scopes
    public function scopePendientes($query)
    {
        return $query->where('estado', 'pendiente');
    }

    public function scopeAprobadas($query)
    {
        return $query->where('estado', 'aprobada');
    }

    // Métodos auxiliares
    public function getPorcentajeProgresoAttribute()
    {
        $total = $this->tareas->count();
        if ($total === 0)
            return 0;

        $finalizados = $this->tareas->where('estado', 'finalizado')->count();
        return round(($finalizados / $total) * 100, 2);
    }
}
