<?php

namespace App\Http\Controllers;

use App\Mail\TareaFinalizada;
use App\Models\Auditoria;
use App\Models\Comentario;
use App\Models\Empresa;
use App\Models\Evidencia;
use App\Models\Material;
use App\Models\Notificacion;
use App\Models\Tarea;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

class TareaController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $user = auth()->user();
        $query = Tarea::with(['empresa', 'operador']);

        // Base filtering based on role
        if ($user->hasRole('administrador')) {
            // No extra filter
            $empresas = Empresa::all();
        } elseif ($user->hasRole('proveedor')) {
            $query->whereHas('empresa', function ($q) use ($user) {
                $q->whereHas('proveedores', function ($q2) use ($user) {
                    $q2->where('users.id', $user->id);
                });
            });
            $empresas = Empresa::whereHas('proveedores', function ($q) use ($user) {
                $q->where('users.id', $user->id);
            })->get();
        } else {
            // Operador ve solo sus tareas
            $query->where('operador_id', $user->id);
            // Get companies that the operator has tasks with
            $empresas = Empresa::whereHas('tareas', function ($q) use ($user) {
                $q->where('operador_id', $user->id);
            })->get();
        }

        // Sort companies by display name (Commercial Name or Legal Name)
        $empresas = $empresas->sortBy(function ($empresa) {
            return strtolower($empresa->nombre_comercial ?? $empresa->razon_social);
        });

        // Apply filters
        if ($request->filled('empresa_id')) {
            $query->where('empresa_id', $request->empresa_id);
        }

        if ($request->filled('estado')) {
            $query->where('tareas.estado', $request->estado);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('tareas.titulo', 'like', "%{$search}%");
        }

        // Sorting
        if ($request->has('sort_by')) {
            $sortBy = $request->sort_by;
            $order = $request->order == 'desc' ? 'desc' : 'asc';

            if ($sortBy == 'operador') {
                $query
                    ->join('users', 'tareas.operador_id', '=', 'users.id')
                    ->orderBy('users.nombre', $order)
                    ->orderBy('users.apellidos', $order)
                    ->select('tareas.*');
            } elseif ($sortBy == 'empresa') {
                $query
                    ->join('empresas', 'tareas.empresa_id', '=', 'empresas.id')
                    ->orderBy('empresas.razon_social', $order)
                    ->select('tareas.*');
            } else {
                // Validate sortable columns to prevent SQL injection or errors
                $allowedSorts = ['titulo', 'estado', 'fecha_inicio_estimada', 'fecha_fin_estimada', 'created_at'];
                if (in_array($sortBy, $allowedSorts)) {
                    $query->orderBy($sortBy, $order);
                } else {
                    $query->orderBy('created_at', 'desc');
                }
            }
        } else {
            $query->orderBy('created_at', 'desc');
        }

        $tareas = $query->paginate(15)->withQueryString();
        return view('tareas.index', compact('tareas', 'empresas'));
    }

    public function kanban(Request $request)
    {
        $user = auth()->user();
        $query = Tarea::with(['empresa', 'operador']);

        // Base filtering based on role (logic copied from index)
        if ($user->hasRole('administrador')) {
            $empresas = Empresa::all();
        } elseif ($user->hasRole('proveedor')) {
            $query->whereHas('empresa', function ($q) use ($user) {
                $q->whereHas('proveedores', function ($q2) use ($user) {
                    $q2->where('users.id', $user->id);
                });
            });
            $empresas = Empresa::whereHas('proveedores', function ($q) use ($user) {
                $q->where('users.id', $user->id);
            })->get();
        } else {
            $query->where('operador_id', $user->id);
            $empresas = Empresa::whereHas('tareas', function ($q) use ($user) {
                $q->where('operador_id', $user->id);
            })->get();
        }

        $empresas = $empresas->sortBy(function ($empresa) {
            return strtolower($empresa->nombre_comercial ?? $empresa->razon_social);
        });

        if ($request->filled('empresa_id')) {
            $query->where('empresa_id', $request->empresa_id);
        }

        // Search filter
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where('titulo', 'like', "%{$search}%");
        }

        $tareas = $query->orderBy('orden', 'asc')->orderBy('created_at', 'desc')->get();

        $kanbanData = [
            'pendiente' => $tareas->where('estado', 'pendiente'),
            'en_progreso' => $tareas->where('estado', 'en_progreso'),
            'finalizado' => $tareas->where('estado', 'finalizado'),
        ];

        return view('tareas.kanban', compact('kanbanData', 'empresas'));
    }

    public function updateOrder(Request $request)
    {
        // Here we could implement reordering within the same column
        // For now, it returns success to allow frontend interactions if implemented
        return response()->json(['success' => true]);
    }

    public function show(Tarea $tarea)
    {
        $tarea->load(['empresa.proveedor', 'operador', 'comentarios.user', 'evidencias.user', 'materiales.user']);

        if (request()->ajax()) {
            return view('tareas.partials.modal_content', compact('tarea'));
        }

        $auditorias = Auditoria::with('user')
            ->where('tabla_afectada', 'tareas')
            ->where('registro_id', $tarea->id)
            ->latest()
            ->get();

        return view('tareas.show', compact('tarea', 'auditorias'));
    }

    public function edit(Tarea $tarea)
    {
        $this->authorize('update', $tarea);

        $operadores = \App\Models\User::role('operador')->where('estado', 'activo')->get();
        $prioridades = [
            'baja' => 'Baja',
            'media' => 'Media',
            'alta' => 'Alta',
            'urgente' => 'Urgente'
        ];
        $estados = [
            'pendiente' => 'Pendiente',
            'en_progreso' => 'En Progreso',
            'finalizado' => 'Finalizado',
            'cancelado' => 'Cancelado'
        ];

        return view('tareas.edit', compact('tarea', 'operadores', 'prioridades', 'estados'));
    }

    public function update(Request $request, Tarea $tarea)
    {
        $this->authorize('update', $tarea);

        $request->validate([
            'titulo' => 'required|string|max:255',
            'descripcion' => 'required|string',
            'fecha_inicio_estimada' => 'required|date',
            'fecha_fin_estimada' => 'required|date|after_or_equal:fecha_inicio_estimada',
            'prioridad' => 'required|in:baja,media,alta,urgente',
            'operador_id' => 'required|exists:users,id',
        ]);

        $datosAnteriores = $tarea->toArray();

        $tarea->update([
            'titulo' => $request->titulo,
            'descripcion' => $request->descripcion,
            'fecha_inicio_estimada' => $request->fecha_inicio_estimada,
            'fecha_fin_estimada' => $request->fecha_fin_estimada,
            'prioridad' => $request->prioridad,
            'operador_id' => $request->operador_id,
        ]);

        Auditoria::registrar('update', 'tareas', $tarea->id, $datosAnteriores, $tarea->fresh()->toArray());

        return redirect()
            ->route('tareas.show', $tarea)
            ->with('success', 'Tarea actualizada exitosamente.');
    }

    public function destroy(Tarea $tarea)
    {
        $this->authorize('delete', $tarea);

        $datosAnteriores = $tarea->toArray();
        $tarea->delete();

        Auditoria::registrar('delete', 'tareas', $tarea->id, $datosAnteriores, null);

        if (request()->ajax() || request()->wantsJson()) {
            return response()->json(['success' => true]);
        }

        return redirect()
            ->route('tareas.index')
            ->with('success', 'Tarea eliminada exitosamente.');
    }

    public function updateStatus(Request $request, Tarea $tarea)
    {
        $this->authorize('updateStatus', $tarea);

        $request->validate([
            'estado' => 'required|in:pendiente,en_progreso,finalizado,cancelado',
            'evidencia' => 'nullable|file|max:10240|mimes:pdf,jpg,jpeg,png,doc,docx,xls,xlsx,zip,rar',
        ]);

        $datosAnteriores = $tarea->toArray();
        $estadoAnterior = $tarea->estado;

        $updateData = ['estado' => $request->estado];

        if ($request->estado === 'en_progreso' && $tarea->estado === 'pendiente') {
            $updateData['fecha_inicio_real'] = now();
        } elseif ($request->estado === 'finalizado') {
            $updateData['fecha_fin_real'] = now();
            // Always update observaciones_finales if present in request
            if ($request->has('observaciones_finales')) {
                $updateData['observaciones_finales'] = $request->observaciones_finales;
            }
        }

        $tarea->update($updateData);

        // Procesar evidencia si se subió
        if ($request->hasFile('evidencia')) {
            $archivo = $request->file('evidencia');
            $count = $tarea->evidencias()->count() + 1;
            $extension = $archivo->getClientOriginalExtension();
            $nombreSecuencial = 'evidencia' . str_pad($count, 3, '0', STR_PAD_LEFT) . '.' . $extension;

            while (Storage::disk('public')->exists('evidencias/' . $tarea->id . '/' . $nombreSecuencial)) {
                $count++;
                $nombreSecuencial = 'evidencia' . str_pad($count, 3, '0', STR_PAD_LEFT) . '.' . $extension;
            }

            $rutaArchivo = $archivo->storeAs('evidencias/' . $tarea->id, $nombreSecuencial, 'public');

            $evidencia = Evidencia::create([
                'tarea_id' => $tarea->id,
                'user_id' => auth()->user()->id,
                'nombre_archivo' => $nombreSecuencial,
                'ruta_archivo' => $rutaArchivo,
                'tipo_archivo' => $archivo->getMimeType(),
                'tamaño_archivo' => $archivo->getSize(),
                'descripcion' => 'Evidencia de finalización',
            ]);

            $this->notificarNuevaEvidencia($tarea, $evidencia);
        }

        // Enviar correo si se finaliza
        if ($request->estado === 'finalizado' && $estadoAnterior != 'finalizado') {
            try {
                $tarea->load('empresa.proveedores');
                foreach ($tarea->empresa->proveedores as $proveedor) {
                    if ($proveedor->email) {
                        Mail::to($proveedor->email)->send(new TareaFinalizada($tarea));
                    }
                }
            } catch (\Exception $e) {
                // Silent fail for email
            }
        }

        // Crear notificaciones
        $this->crearNotificacionesCambioEstado($tarea, $estadoAnterior, $request->estado);

        Auditoria::registrar('update_status', 'tareas', $tarea->id, $datosAnteriores, $tarea->fresh()->toArray());

        if ($request->wantsJson()) {
            return response()->json(['success' => true, 'message' => 'Estado actualizado a ' . ucfirst(str_replace('_', ' ', $request->estado))]);
        }

        return redirect()
            ->route('tareas.show', $tarea)
            ->with('success', 'Estado de la tarea actualizado exitosamente.');
    }

    public function addComment(Request $request, Tarea $tarea)
    {
        $this->authorize('comment', $tarea);

        $request->validate([
            'comentario' => 'required|string',
        ]);

        $comentario = Comentario::create([
            'tarea_id' => $tarea->id,
            'user_id' => auth()->user()->id,
            'comentario' => $request->comentario,
        ]);

        // Notificar a stakeholders
        $this->notificarNuevoComentario($tarea, $comentario);

        Auditoria::registrar('add_comment', 'comentarios', $comentario->id, null, $comentario->toArray());

        if ($request->ajax() || $request->wantsJson()) {
            return response()->json(['success' => true]);
        }

        return redirect()
            ->route('tareas.show', $tarea)
            ->with('success', 'Comentario añadido exitosamente.');
    }

    public function uploadEvidence(Request $request, Tarea $tarea)
    {
        $this->authorize('uploadEvidence', $tarea);

        $request->validate([
            'archivo' => 'required|file|max:10240|mimes:pdf,jpg,jpeg,png,doc,docx,xls,xlsx',
            'descripcion' => 'nullable|string',
        ]);

        $archivo = $request->file('archivo');

        // Generar nombre secuencial
        $count = $tarea->evidencias()->count() + 1;
        $extension = $archivo->getClientOriginalExtension();
        // Formato: evidencia001.ext
        $nombreSecuencial = 'evidencia' . str_pad($count, 3, '0', STR_PAD_LEFT) . '.' . $extension;

        // Para evitar colisiones si se borran archivos intermedios, verificamos existencia
        while (Storage::disk('public')->exists('evidencias/' . $tarea->id . '/' . $nombreSecuencial)) {
            $count++;
            $nombreSecuencial = 'evidencia' . str_pad($count, 3, '0', STR_PAD_LEFT) . '.' . $extension;
        }

        // Guardar en carpeta específica de la tarea
        $rutaArchivo = $archivo->storeAs('evidencias/' . $tarea->id, $nombreSecuencial, 'public');

        $evidencia = Evidencia::create([
            'tarea_id' => $tarea->id,
            'user_id' => auth()->user()->id,
            'nombre_archivo' => $nombreSecuencial,  // Guardamos el nombre secuencial
            'ruta_archivo' => $rutaArchivo,
            'tipo_archivo' => $archivo->getMimeType(),
            'tamaño_archivo' => $archivo->getSize(),
            'descripcion' => $request->descripcion,
        ]);

        // Notificar nueva evidencia
        $this->notificarNuevaEvidencia($tarea, $evidencia);

        Auditoria::registrar('upload_evidence', 'evidencias', $evidencia->id, null, $evidencia->toArray());

        return redirect()
            ->route('tareas.show', $tarea)
            ->with('success', 'Evidencia subida exitosamente.');
    }

    public function deleteEvidence(Evidencia $evidencia)
    {
        $this->authorize('delete', $evidencia);

        $tarea = $evidencia->tarea;

        // Eliminar archivo físico
        if (Storage::disk('public')->exists($evidencia->ruta_archivo)) {
            Storage::disk('public')->delete($evidencia->ruta_archivo);
        }

        $datosAnteriores = $evidencia->toArray();
        $evidencia->delete();

        Auditoria::registrar('delete_evidence', 'evidencias', $evidencia->id, $datosAnteriores, null);

        return redirect()
            ->route('tareas.show', $tarea)
            ->with('success', 'Evidencia eliminada exitosamente.');
    }

    public function downloadEvidence(Evidencia $evidencia)
    {
        $this->authorize('view', $evidencia->tarea);

        if (!Storage::disk('public')->exists($evidencia->ruta_archivo)) {
            abort(404, 'Archivo no encontrado');
        }

        return Storage::disk('public')->download($evidencia->ruta_archivo, $evidencia->nombre_archivo);
    }

    // Gestión de Materiales
    public function uploadMaterial(Request $request, Tarea $tarea)
    {
        $this->authorize('uploadMaterial', $tarea);

        $request->validate([
            'archivo' => 'required|file|max:10240',  // 10MB max
            'descripcion' => 'nullable|string',
        ]);

        $archivo = $request->file('archivo');
        $count = $tarea->materiales()->count() + 1;
        $extension = $archivo->getClientOriginalExtension();
        $nombreFinal = 'material' . str_pad($count, 3, '0', STR_PAD_LEFT) . '.' . $extension;
        while (Storage::disk('public')->exists('materiales/' . $tarea->id . '/' . $nombreFinal)) {
            $count++;
            $nombreFinal = 'material' . str_pad($count, 3, '0', STR_PAD_LEFT) . '.' . $extension;
        }

        $rutaArchivo = $archivo->storeAs('materiales/' . $tarea->id, $nombreFinal, 'public');

        $material = Material::create([
            'tarea_id' => $tarea->id,
            'user_id' => auth()->user()->id,
            'nombre_archivo' => $nombreFinal,
            'ruta_archivo' => $rutaArchivo,
            'tipo_archivo' => $archivo->getMimeType(),
            'tamaño_archivo' => $archivo->getSize(),
            'descripcion' => $request->descripcion,
        ]);

        Auditoria::registrar('upload_material', 'materiales', $material->id, null, $material->toArray());

        return redirect()
            ->route('tareas.show', $tarea)
            ->with('success', 'Material de apoyo subido exitosamente.');
    }

    public function deleteMaterial(Material $material)
    {
        $this->authorize('delete', $material);

        $tarea = $material->tarea;

        if (Storage::disk('public')->exists($material->ruta_archivo)) {
            Storage::disk('public')->delete($material->ruta_archivo);
        }

        $datosAnteriores = $material->toArray();
        $material->delete();

        Auditoria::registrar('delete_material', 'materiales', $material->id, $datosAnteriores, null);

        return redirect()
            ->route('tareas.show', $tarea)
            ->with('success', 'Material eliminado exitosamente.');
    }

    public function downloadMaterial(Material $material)
    {
        $this->authorize('download', $material);

        if (!Storage::disk('public')->exists($material->ruta_archivo)) {
            abort(404, 'Archivo no encontrado');
        }

        return Storage::disk('public')->download($material->ruta_archivo, $material->nombre_archivo);
    }

    private function crearNotificacionesCambioEstado(Tarea $tarea, $estadoAnterior, $estadoNuevo)
    {
        $usuario = auth()->user();
        $mensaje = "La tarea '{$tarea->titulo}' cambió de estado de '{$estadoAnterior}' a '{$estadoNuevo}'";

        // Notificar al operador (si no es quien realizó el cambio)
        if ($tarea->operador_id && $tarea->operador_id != $usuario->id) {
            Notificacion::create([
                'user_id' => $tarea->operador_id,
                'titulo' => 'Cambio de estado en tarea',
                'mensaje' => $mensaje,
                'tipo' => 'info',
            ]);
        }

        // Notificar a los proveedores (si no son quienes realizaron el cambio)
        foreach ($tarea->empresa->proveedores as $proveedor) {
            if ($proveedor->id != $usuario->id) {
                Notificacion::create([
                    'user_id' => $proveedor->id,
                    'titulo' => 'Cambio de estado en tarea',
                    'mensaje' => $mensaje,
                    'tipo' => 'info',
                ]);
            }
        }

        // Notificar a administradores (si no son quienes realizaron el cambio)
        $admins = \App\Models\User::role('administrador')->get();
        foreach ($admins as $admin) {
            if ($admin->id != $usuario->id) {
                Notificacion::create([
                    'user_id' => $admin->id,
                    'titulo' => 'Cambio de estado en tarea',
                    'mensaje' => $mensaje,
                    'tipo' => 'info',
                ]);
            }
        }
    }

    private function notificarNuevoComentario(Tarea $tarea, Comentario $comentario)
    {
        $usuario = auth()->user();
        $mensaje = "Nuevo comentario en la tarea '{$tarea->titulo}' por {$usuario->nombre_completo}";

        // Notificar a los proveedores asignados (si no son quienes comentaron)
        foreach ($tarea->empresa->proveedores as $proveedor) {
            if ($proveedor->id != $usuario->id) {
                Notificacion::create([
                    'user_id' => $proveedor->id,
                    'titulo' => 'Nuevo comentario',
                    'mensaje' => $mensaje,
                    'tipo' => 'info',
                ]);
            }
        }

        // Notificar al operador (si no es quien comentó)
        if ($tarea->operador_id != $usuario->id) {
            Notificacion::create([
                'user_id' => $tarea->operador_id,
                'titulo' => 'Nuevo comentario',
                'mensaje' => $mensaje,
                'tipo' => 'info',
            ]);
        }
    }

    private function notificarNuevaEvidencia(Tarea $tarea, Evidencia $evidencia)
    {
        $usuario = auth()->user();
        $mensaje = "Nueva evidencia subida en la tarea '{$tarea->titulo}' por {$usuario->nombre_completo}";

        // Notificar al operador (si no es quien subió la evidencia)
        if ($tarea->operador_id && $tarea->operador_id != $usuario->id) {
            Notificacion::create([
                'user_id' => $tarea->operador_id,
                'titulo' => 'Nueva evidencia',
                'mensaje' => $mensaje,
                'tipo' => 'success',
            ]);
        }

        // Notificar a los proveedores asignados (si no son quienes subieron la evidencia)
        foreach ($tarea->empresa->proveedores as $proveedor) {
            if ($proveedor->id != $usuario->id) {
                Notificacion::create([
                    'user_id' => $proveedor->id,
                    'titulo' => 'Nueva evidencia',
                    'mensaje' => $mensaje,
                    'tipo' => 'success',
                ]);
            }
        }

        // Notificar a administradores (si no son quienes subieron la evidencia)
        $admins = \App\Models\User::role('administrador')->get();
        foreach ($admins as $admin) {
            if ($admin->id != $usuario->id) {
                Notificacion::create([
                    'user_id' => $admin->id,
                    'titulo' => 'Nueva evidencia',
                    'mensaje' => $mensaje,
                    'tipo' => 'success',
                ]);
            }
        }
    }

    public function modalContent(Tarea $tarea)
    {
        // Use basic view policy or check if user can view the task
        // $this->authorize('view', $tarea);
        // Note: Creating a separate policy check or reusing view might be needed depending on strictness

        $tarea->load(['empresa', 'operador', 'comentarios.user', 'evidencias']);
        return view('tareas.partials.modal_content', compact('tarea'));
    }

    public function modalEditContent(Tarea $tarea)
    {
        $this->authorize('update', $tarea);
        $tarea->load('materiales');

        $operadores = \App\Models\User::role('operador')->where('estado', 'activo')->get();
        $prioridades = [
            'baja' => 'Baja',
            'media' => 'Media',
            'alta' => 'Alta',
            'urgente' => 'Urgente'
        ];
        $estados = [
            'pendiente' => 'Pendiente',
            'en_progreso' => 'En Progreso',
            'finalizado' => 'Finalizado',
            'cancelado' => 'Cancelado'
        ];

        return view('tareas.partials.edit_modal_content', compact('tarea', 'operadores', 'prioridades', 'estados'));
    }
}
