<?php

namespace App\Http\Controllers;

use App\Models\Empresa;
use App\Models\Tarea;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade as PDF;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\ReporteMensualExport;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class ReporteController extends Controller
{
    public function __construct()
    {
        // Permitir acceso a proveedor y administrador
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            if (!auth()->user()->hasAnyRole(['proveedor', 'administrador'])) {
                abort(403, 'No tiene permisos para acceder a esta sección.');
            }
            return $next($request);
        });
    }

    public function index()
    {
        $user = auth()->user();
        
        $query = Empresa::whereNotIn('estado', ['inactiva', 'retirada', 'rechazada']);

        if ($user->hasRole('administrador')) {
            // Administrador ve todas las empresas vigentes
            $empresas = $query->orderBy('razon_social')->get();
        } else {
            // Proveedor ve sus empresas asignadas
            $empresas = $query->whereHas('proveedores', function($q) use ($user) {
                $q->where('user_id', $user->id);
            })->orderBy('razon_social')->get();
        }
        
        return view('reportes.index', compact('empresas'));
    }

    public function exportar(Request $request)
    {
        $request->validate([
            'mes' => 'required|integer|min:1|max:12',
            'anio' => 'required|integer|min:2020|max:'.(date('Y')+1),
            'empresas' => 'required|array',
            'empresas.*' => 'exists:empresas,id',
            'formato' => 'required|in:excel,pdf'
        ]);

        $mes = $request->mes;
        $anio = $request->anio;
        $empresaIds = $request->empresas;
        $user = auth()->user();

        // Validar que las empresas pertenezcan al proveedor o permitir todas si es admin
        $query = Empresa::whereIn('id', $empresaIds)->whereNotIn('estado', ['inactiva', 'retirada', 'rechazada']);

        if ($user->hasRole('administrador')) {
            $empresasValidas = $query->pluck('id');
        } else {
            $empresasValidas = $query->whereHas('proveedores', function($q) use ($user) {
                    $q->where('user_id', $user->id);
                })
                ->pluck('id');
        }

        if ($empresasValidas->isEmpty()) {
            return back()->withErrors(['empresas' => 'Debe seleccionar al menos una empresa válida.']);
        }

        // Obtener tareas
        $tareas = Tarea::whereIn('empresa_id', $empresasValidas)
            ->where('estado', 'finalizado')
            ->whereYear('fecha_inicio_estimada', $anio)
            ->whereMonth('fecha_inicio_estimada', $mes)
            ->with(['empresa', 'operador'])
            ->orderBy('empresa_id')
            ->orderBy('fecha_inicio_estimada')
            ->get();

        $data = [
            'tareas' => $tareas,
            'mes' => $mes,
            'anio' => $anio,
            'fecha_generacion' => now()->format('d/m/Y H:i'),
            'proveedor' => $user->nombre_completo,
            'total_tareas' => $tareas->count(),
            'tareas_completadas' => $tareas->where('estado', 'finalizado')->count(),
            'tareas_pendientes' => $tareas->where('estado', '!=', 'finalizado')->count(),
        ];

        if ($request->formato === 'pdf') {
            $pdf = PDF::loadView('reportes.pdf', $data);
            $pdf->setPaper('a4', 'landscape');
            return $pdf->download("reporte_mensual_{$mes}_{$anio}.pdf");
        } else {
            return Excel::download(new ReporteMensualExport($data), "reporte_mensual_{$mes}_{$anio}.xlsx");
        }
    }
}
