<?php
// app/Http/Controllers/Proveedor/DashboardController.php

namespace App\Http\Controllers\Proveedor;

use App\Http\Controllers\Controller;
use App\Models\Empresa;
use App\Models\Tarea;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:proveedor']);
    }

    public function index()
    {
        $user = auth()->user();

        // Helper query closure
        $whereHasProveedores = function ($q) use ($user) {
            $q->whereHas('proveedores', function ($subQ) use ($user) {
                $subQ->where('user_id', $user->id);
            });
        };
        
        // Helper query closure for Tareas
        $whereHasEmpresaProveedores = function ($q) use ($user) {
            $q->whereHas('empresa', function ($empresaQ) use ($user) {
                $empresaQ->whereHas('proveedores', function ($provQ) use ($user) {
                    $provQ->where('user_id', $user->id);
                });
            });
        };

        $stats = [
            'empresas_creadas' => Empresa::whereHas('proveedores', function ($q) use ($user) {
                $q->where('user_id', $user->id);
            })->count(),
            'empresas_aprobadas' => Empresa::whereHas('proveedores', function ($q) use ($user) {
                $q->where('user_id', $user->id);
            })
                ->where('estado', 'aprobada')
                ->count(),
            'empresas_pendientes' => Empresa::whereHas('proveedores', function ($q) use ($user) {
                $q->where('user_id', $user->id);
            })
                ->where('estado', 'pendiente')
                ->count(),
            'empresas_rechazadas' => Empresa::whereHas('proveedores', function ($q) use ($user) {
                $q->where('user_id', $user->id);
            })
                ->where('estado', 'rechazada')
                ->count(),
            'tareas_total' => Tarea::where($whereHasEmpresaProveedores)->count(),
            'tareas_finalizadas' => Tarea::where($whereHasEmpresaProveedores)
                ->where('estado', 'finalizado')->count(),
            'tareas_en_progreso' => Tarea::where($whereHasEmpresaProveedores)
                ->where('estado', 'en_progreso')->count(),
        ];

        // Calcular porcentaje de éxito
        $stats['porcentaje_exito'] = $stats['tareas_total'] > 0
            ? round(($stats['tareas_finalizadas'] / $stats['tareas_total']) * 100, 2)
            : 0;

        $empresasRecientes = Empresa::whereHas('proveedores', function ($q) use ($user) {
                $q->where('user_id', $user->id);
            })
            ->with(['tareas' => function ($q) {
                $q->orderBy('orden');
            }])
            ->orderBy('created_at', 'desc')
            ->take(5)
            ->get();

        $tareasActivas = Tarea::where($whereHasEmpresaProveedores)
            ->where('estado', '!=', 'finalizado')
            ->with(['empresa', 'operador'])
            ->orderBy('prioridad')
            ->orderBy('fecha_fin_estimada')
            ->take(10)
            ->get();

        $tareasRetrasadas = Tarea::where($whereHasEmpresaProveedores)
            ->where('estado', '!=', 'finalizado')
            ->where('fecha_fin_estimada', '<', now())
            ->with(['empresa', 'operador'])
            ->orderBy('fecha_fin_estimada')
            ->get();

        return view('proveedor.dashboard', compact('stats', 'empresasRecientes', 'tareasActivas', 'tareasRetrasadas'));
    }
}
