<?php
// app/Http/Controllers/NotificacionController.php

namespace App\Http\Controllers;

use App\Models\Notificacion;
use App\Models\User;
use Illuminate\Http\Request;

class NotificacionController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index()
    {
        $notificaciones = Notificacion::where('user_id', auth()->user()->id)
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('notificaciones.index', compact('notificaciones'));
    }

    public function show(Notificacion $notificacione)
    {
        // return $notificacione;
        $this->authorize('view', $notificacione);

        // Marcar como leída al verla
        if (!$notificacione->leida) {
            $notificacione->marcarComoLeida();
        }
        $notificacion = $notificacione;
        return view('notificaciones.show', compact('notificacion'));
    }

    public function create()
    {
        $this->authorize('create', Notificacion::class);

        $usuarios = User::where('estado', 'activo')->get();

        return view('notificaciones.create', compact('usuarios'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', Notificacion::class);

        $request->validate([
            'titulo' => 'required|string|max:255',
            'mensaje' => 'required|string',
            'tipo' => 'required|in:info,success,warning,error',
            'users' => 'required|array',
            'users.*' => 'exists:users,id',
        ]);

        foreach ($request->users as $userId) {
            Notificacion::create([
                'user_id' => $userId,
                'titulo' => $request->titulo,
                'mensaje' => $request->mensaje,
                'tipo' => $request->tipo,
            ]);
        }

        return redirect()->route('notificaciones.index')
            ->with('success', 'Notificaciones enviadas exitosamente.');
    }

    public function destroy(Notificacion $notificacione)
    {
        $this->authorize('delete', $notificacione);

        $notificacione->delete();

        return redirect()->route('notificaciones.index')
            ->with('success', 'Notificación eliminada exitosamente.');
    }

    public function markAsRead(Notificacion $notificacion)
    {
        // return response()->json(['success' => $notificacion]);
        $this->authorize('markAsRead', $notificacion);

        // $notificacion->marcarComoLeida();

        $notificacion->update([
            'leida' => true,
            'fecha_lectura' => now(),
        ]);

        return response()->json(['success' => true]);
    }

    public function markAllAsRead()
    {
        Notificacion::where('user_id', auth()->user()->id)
            ->where('leida', false)
            ->update([
                'leida' => true,
                'fecha_lectura' => now(),
            ]);

        return response()->json(['success' => true]);
    }

    public function getUnreadCount()
    {
        $count = Notificacion::where('user_id', auth()->user()->id)
            ->where('leida', false)
            ->count();

        return response()->json(['count' => $count]);
    }

    public function clearAll()
    {
        Notificacion::where('user_id', auth()->user()->id)->delete();

        return redirect()->route('notificaciones.index')
            ->with('success', 'Todas las notificaciones han sido eliminadas.');
    }
}
