<?php

namespace App\Http\Controllers;

use App\Mail\NuevaTareaAsignada;
use App\Models\ApiKey;
use App\Models\Auditoria;
use App\Models\Empresa;
use App\Models\EmpresaRuc;
use App\Models\Notificacion;
use App\Models\Tarea;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;

class EmpresaController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth')->except(['credenciales']);
    }

    public function index(Request $request)
    {
        $user = auth()->user();

        $query = Empresa::with(['proveedores', 'tareas', 'otrosRucs']);

        if ($user->hasRole('administrador')) {
            // Administrador ve todo
        } elseif ($user->hasRole('proveedor')) {
            // Proveedor ve empresas asignadas en la tabla pivote
            $query->whereHas('proveedores', function ($q) use ($user) {
                $q->where('user_id', $user->id);
            });
        } else {
            // Operador ve empresas que tienen tareas asignadas a él
            $query->whereHas('tareas', function ($q) use ($user) {
                $q->where('operador_id', $user->id);
            });
        }

        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where(function ($q) use ($search) {
                $q
                    ->where('razon_social', 'like', "%{$search}%")
                    ->orWhere('nombre_comercial', 'like', "%{$search}%")
                    ->orWhere('ruc', 'like', "%{$search}%");
            });
        }

        // Ordenar por empresas que tienen más tareas pendientes/en progreso primero
        $query->withCount(['tareas as tareas_activas_count' => function ($q) {
            $q->whereIn('estado', ['pendiente', 'en_progreso']);
        }]);

        $empresas = $query
            ->orderBy('tareas_activas_count', 'desc')
            ->orderBy('created_at', 'desc')
            ->paginate(10);
        $empresas->appends($request->all());

        if ($request->ajax()) {
            return view('empresas.partials.list', compact('empresas'))->render();
        }

        $operadores = User::role('operador')->where('estado', 'activo')->get();
        $proveedores = User::role('proveedor')->where('estado', 'activo')->get();

        return view('empresas.index', compact('empresas', 'operadores', 'proveedores'));
    }

    public function credenciales(Request $request)
    {
        $user = auth()->user();
        $query = Empresa::query();

        if ($user) {
            if ($user->hasRole('administrador')) {
                // Administrador ve todo
            } elseif ($user->hasRole('proveedor')) {
                // Proveedor ve empresas asignadas en la tabla pivote
                $query->whereHas('proveedores', function ($q) use ($user) {
                    $q->where('user_id', $user->id);
                });
            } else {
                // Operador ve empresas que tienen tareas asignadas a él
                $query->whereHas('tareas', function ($q) use ($user) {
                    $q->where('operador_id', $user->id);
                });
            }
        }
        // Si no hay usuario logueado, ve todas las empresas (público)

        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where(function ($q) use ($search) {
                $q
                    ->where('razon_social', 'like', "%{$search}%")
                    ->orWhere('nombre_comercial', 'like', "%{$search}%")
                    ->orWhere('ruc', 'like', "%{$search}%");
            });
        }

        $sortBy = $request->input('sort_by', 'razon_social');
        $sortOrder = $request->input('sort_order', 'asc');

        // Validar columnas permitidas
        $allowedSorts = ['ruc', 'razon_social', 'nombre_comercial'];
        if (!in_array($sortBy, $allowedSorts)) {
            $sortBy = 'razon_social';
        }
        if (!in_array($sortOrder, ['asc', 'desc'])) {
            $sortOrder = 'asc';
        }

        $empresas = $query
            ->orderBy($sortBy, $sortOrder)
            ->paginate(15);
        $empresas->appends($request->all());

        // Obtener API Key válida
        $apiKey = ApiKey::where('is_active', true)
            ->where(function ($query) {
                $query
                    ->whereNull('expires_at')
                    ->orWhere('expires_at', '>', Carbon::now());
            })
            ->latest()
            ->first();

        $apiKeyValue = $apiKey ? $apiKey->key : '';

        if ($request->ajax()) {
            return view('empresas.partials.credenciales_table', compact('empresas', 'sortBy', 'sortOrder'))->render();
        }

        return view('empresas.credenciales', compact('empresas', 'apiKeyValue', 'sortBy', 'sortOrder'));
    }

    public function create()
    {
        $this->authorize('create', Empresa::class);
        $operadores = User::role('operador')->where('estado', 'activo')->get();
        $proveedores = User::role('proveedor')->where('estado', 'activo')->get();  // Obtener proveedores para selección

        return view('empresas.create', compact('operadores', 'proveedores'));
    }

    public function store(Request $request)
    {
        $this->authorize('create', Empresa::class);

        $request->validate([
            'ruc' => 'required|string|max:11',
            'razon_social' => 'required|string|max:255',
            'nombre_comercial' => 'nullable|string|max:255',
            'sitio_web' => 'nullable|url',
            'serie' => 'nullable|string|max:3',
            'usuario_sol' => 'nullable|string|max:255',
            'clave_sol' => 'nullable|string|max:255',
            'usuario_secundario_dni' => 'nullable|string|max:8',
            'usuario_secundario_user' => 'nullable|string|max:255',
            'usuario_secundario_clave' => 'nullable|string|max:255',
            'clave_certificado' => 'nullable|string|max:255',
            'descripcion' => 'nullable|string',
            'fecha_solicitud' => 'nullable|date',
            'modo_pago' => 'required|in:mensual,anual,gratuito',
            'proveedores' => 'nullable|array',
            'proveedores.*' => 'exists:users,id',
            'tareas' => 'nullable|array',
            'tareas.*.titulo' => 'required_with:tareas|string|max:255',
            'tareas.*.descripcion' => 'required_with:tareas|string',
            'tareas.*.fecha_inicio_estimada' => 'required_with:tareas|date',
            'tareas.*.fecha_fin_estimada' => 'required_with:tareas|date|after_or_equal:tareas.*.fecha_inicio_estimada',
            'tareas.*.prioridad' => 'required_with:tareas|in:baja,media,alta,urgente',
            'tareas.*.operador_id' => 'required_with:tareas|exists:users,id',
        ]);

        DB::beginTransaction();

        try {
            // Determinar el proveedor_id principal (creador o actual usuario si es proveedor)
            $proveedorId = auth()->user()->hasRole('proveedor') ? auth()->user()->id : null;

            $empresa = Empresa::create([
                'ruc' => $request->ruc,
                'razon_social' => $request->razon_social,
                'nombre_comercial' => $request->nombre_comercial,
                'sitio_web' => $request->sitio_web,
                'serie' => $request->serie,
                'modo_pago' => $request->modo_pago,
                'usuario_sol' => $request->usuario_sol,
                'clave_sol' => $request->clave_sol,
                'usuario_secundario_dni' => $request->usuario_secundario_dni,
                'usuario_secundario_user' => $request->usuario_secundario_user,
                'usuario_secundario_clave' => $request->usuario_secundario_clave,
                'clave_certificado' => $request->clave_certificado,
                'descripcion' => $request->descripcion,
                'fecha_solicitud' => $request->fecha_solicitud ?? now(),
                'proveedor_id' => $proveedorId,
                'estado' => 'aprobada',  // Auto-aprobado por defecto
                'aprobado_por' => auth()->user()->id,
                'fecha_aprobacion' => now(),
            ]);

            // Asignar proveedores en la tabla pivote

            // 1. Si el creador es proveedor, asignarse automáticamente
            if (auth()->user()->hasRole('proveedor')) {
                $empresa->proveedores()->attach(auth()->user()->id);
            }

            // 2. Si es administrador, permitir asignar otros proveedores seleccionados
            if (auth()->user()->hasRole('administrador') && $request->has('proveedores')) {
                $ids = array_map('intval', (array) $request->input('proveedores', []));
                $ids = array_values(array_filter(array_unique($ids), function ($v) {
                    return $v > 0 && $v != auth()->id();  // Avoid duplicate attach if logic 1 already ran
                }));

                if (!empty($ids)) {
                    $empresa->proveedores()->attach($ids);  // attach matches 'syncWithoutDetaching' behavior for new records
                    // using attach on new record is safe.
                    // But since we might have attached auth()->id() above, ensure we don't error on duplicate entry if user selected themselves too.
                    // 'attach' typically throws error on duplicate PK in pivot if not handled.
                    // Better use syncWithoutDetaching for the batch.
                }
            }

            // Crear tareas iniciales si existen
            if ($request->has('tareas')) {
                foreach ($request->tareas as $tareaData) {
                    $empresa->tareas()->create([
                        'titulo' => $tareaData['titulo'],
                        'descripcion' => $tareaData['descripcion'],
                        'fecha_inicio_estimada' => $tareaData['fecha_inicio_estimada'],
                        'fecha_fin_estimada' => $tareaData['fecha_fin_estimada'],
                        'prioridad' => $tareaData['prioridad'],
                        'operador_id' => $tareaData['operador_id'],
                        'estado' => 'pendiente',
                        'creado_por' => auth()->user()->id,
                    ]);
                }
            }

            // Notificar a administradores sobre nueva empresa
            $administradores = User::role('administrador')->get();
            foreach ($administradores as $admin) {
                if ($admin->id != auth()->id()) {
                    Notificacion::create([
                        'user_id' => $admin->id,
                        'titulo' => 'Nueva Empresa Registrada',
                        'mensaje' => "Se ha registrado la empresa {$empresa->razon_social}",
                        'tipo' => 'info',
                        'leido' => false,
                    ]);
                }
            }

            // Notificar a proveedores asignados (si no son quienes crearon la empresa)
            if ($request->has('proveedores')) {
                foreach ($request->proveedores as $proveedorId) {
                    if ($proveedorId != auth()->id()) {
                        Notificacion::create([
                            'user_id' => $proveedorId,
                            'titulo' => 'Nueva Empresa Asignada',
                            'mensaje' => "Se te ha asignado la empresa {$empresa->razon_social}",
                            'tipo' => 'info',
                            'leido' => false,
                        ]);
                    }
                }
            }

            // Si se asignaron tareas, notificar a operadores
            if ($request->has('tareas')) {
                $operadoresIds = collect($request->tareas)->pluck('operador_id')->unique();
                foreach ($operadoresIds as $operadorId) {
                    Notificacion::create([
                        'user_id' => $operadorId,
                        'titulo' => 'Nueva Tarea Asignada',
                        'mensaje' => "Se te ha asignado una nueva tarea en la empresa {$empresa->razon_social}",
                        'tipo' => 'info',
                        'leido' => false,
                    ]);
                }
            }

            DB::commit();

            Auditoria::registrar('create', 'empresas', $empresa->id, null, $empresa->toArray());

            return redirect()
                ->route('empresas.index')
                ->with('success', 'Empresa registrada exitosamente.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()
                ->with('error', 'Error al registrar la empresa: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function show(Empresa $empresa)
    {
        $empresa->load([
            'proveedor',
            'proveedores',
            'aprobadoPor',
            'tareas' => function ($query) {
                $query->orderBy('orden');
            },
            'tareas.operador'
        ]);

        return view('empresas.show', compact('empresa'));
    }

    public function storeRuc(Request $request, Empresa $empresa)
    {
        $this->authorize('update', $empresa);

        $request->validate([
            'ruc' => 'required|string|max:11',
            'razon_social' => 'required|string|max:255',
            'serie' => 'nullable|string|max:3',
            'usuario_sol' => 'nullable|string|max:255',
            'clave_sol' => 'nullable|string|max:255',
            'usuario_secundario_dni' => 'nullable|string|max:8',
            'usuario_secundario_user' => 'nullable|string|max:255',
            'usuario_secundario_clave' => 'nullable|string|max:255',
            'clave_certificado' => 'nullable|string|max:255',
        ]);

        if (EmpresaRuc::where('empresa_id', $empresa->id)->where('ruc', $request->ruc)->exists()) {
            return back()->with('error', 'Ese RUC ya está asignado a esta empresa.');
        }

        $empresa->otrosRucs()->create([
            'ruc' => $request->ruc,
            'razon_social' => $request->razon_social,
            'serie' => $request->serie,
            'usuario_sol' => $request->usuario_sol,
            'clave_sol' => $request->clave_sol,
            'usuario_secundario_dni' => $request->usuario_secundario_dni,
            'usuario_secundario_user' => $request->usuario_secundario_user,
            'usuario_secundario_clave' => $request->usuario_secundario_clave,
            'clave_certificado' => $request->clave_certificado,
        ]);

        return redirect()->route('empresas.index')->with('success', 'RUC agregado a la empresa.');
    }

    public function edit(Empresa $empresa)
    {
        $this->authorize('update', $empresa);
        $operadores = User::role('operador')->where('estado', 'activo')->get();
        $proveedores = User::role('proveedor')->where('estado', 'activo')->get();
        return view('empresas.edit', compact('empresa', 'operadores', 'proveedores'));
    }

    public function update(Request $request, Empresa $empresa)
    {
        $this->authorize('update', $empresa);

        $datosAnteriores = $empresa->toArray();

        $request->validate([
            'ruc' => 'required|string|max:11',
            'razon_social' => 'required|string|max:255',
            'nombre_comercial' => 'nullable|string|max:255',
            'sitio_web' => 'nullable|url',
            'serie' => 'nullable|string|max:3',
            'modo_pago' => 'required|in:mensual,anual,gratuito',
            'usuario_sol' => 'nullable|string|max:255',
            'clave_sol' => 'nullable|string|max:255',
            'usuario_secundario_dni' => 'nullable|string|max:8',
            'usuario_secundario_user' => 'nullable|string|max:255',
            'usuario_secundario_clave' => 'nullable|string|max:255',
            'clave_certificado' => 'nullable|string|max:255',
            'descripcion' => 'nullable|string',
            'fecha_solicitud' => 'nullable|date',
            'proveedores' => 'nullable|array',
            'proveedores.*' => 'exists:users,id',
            // 'estado' validation removed from here as it is handled in updateStatus or ignored if not present
        ]);

        $data = [
            'ruc' => $request->ruc,
            'razon_social' => $request->razon_social,
            'nombre_comercial' => $request->nombre_comercial,
            'sitio_web' => $request->sitio_web,
            'serie' => $request->serie,
            'modo_pago' => $request->modo_pago,
            'usuario_sol' => $request->usuario_sol,
            'clave_sol' => $request->clave_sol,
            'usuario_secundario_dni' => $request->usuario_secundario_dni,
            'usuario_secundario_user' => $request->usuario_secundario_user,
            'usuario_secundario_clave' => $request->usuario_secundario_clave,
            'clave_certificado' => $request->clave_certificado,
            'descripcion' => $request->descripcion,
            'fecha_solicitud' => $request->fecha_solicitud ?? $empresa->fecha_solicitud,
        ];

        // Status update removed from here to be handled by updateStatus method
        
        $empresa->update($data);

        if (auth()->user()->hasAnyRole(['administrador', 'proveedor'])) {
            $ids = array_map('intval', (array) $request->input('proveedores', []));
            if (auth()->user()->hasRole('proveedor')) {
                $ids[] = (int) auth()->id();
            }
            $ids = array_values(array_filter(array_unique($ids), function ($v) {
                return $v > 0;
            }));
            $empresa->proveedores()->sync($ids);
        }

        Auditoria::registrar('update', 'empresas', $empresa->id, $datosAnteriores, $empresa->fresh()->toArray());

        return redirect()
            ->route('empresas.index')
            ->with('success', 'Empresa actualizada exitosamente.');
    }

    public function updateStatus(Request $request, Empresa $empresa)
    {
        $this->authorize('update', $empresa);

        if (!auth()->user()->hasRole('administrador') && !auth()->user()->hasRole('proveedor')) {
            abort(403, 'No tiene permisos para cambiar el estado.');
        }

        $request->validate([
            'estado' => 'required|in:pendiente,aprobada,rechazada,inactiva,retirada',
        ]);

        $datosAnteriores = $empresa->toArray();

        $empresa->update([
            'estado' => $request->estado,
            'aprobado_por' => ($request->estado == 'aprobada') ? auth()->user()->id : $empresa->aprobado_por,
            'fecha_aprobacion' => ($request->estado == 'aprobada') ? now() : $empresa->fecha_aprobacion,
        ]);

        Auditoria::registrar('update_status', 'empresas', $empresa->id, $datosAnteriores, $empresa->fresh()->toArray());

        if ($request->ajax()) {
            return response()->json(['success' => true, 'message' => 'Estado actualizado correctamente.']);
        }

        return redirect()->route('empresas.index')->with('success', 'Estado actualizado correctamente.');
    }

    public function approve(Empresa $empresa)
    {
        $this->authorize('approve', $empresa);

        $datosAnteriores = $empresa->toArray();

        $empresa->update([
            'estado' => 'aprobada',
            'aprobado_por' => auth()->user()->id,
            'fecha_aprobacion' => now(),
        ]);

        // Notificar a los proveedores asignados
        foreach ($empresa->proveedores as $proveedor) {
            Notificacion::create([
                'user_id' => $proveedor->id,
                'titulo' => 'Empresa aprobada',
                'mensaje' => "La empresa '{$empresa->razon_social}' ha sido aprobada.",
                'tipo' => 'success',
            ]);
        }

        // Notificar a operadores asignados
        $operadores = User::whereIn('id', $empresa->tareas->pluck('operador_id')->unique())->get();
        foreach ($operadores as $operador) {
            Notificacion::create([
                'user_id' => $operador->id,
                'titulo' => 'Nuevas tareas asignadas',
                'mensaje' => "Se te han asignado tareas de la empresa: {$empresa->razon_social}",
                'tipo' => 'info',
            ]);
        }

        Auditoria::registrar('approve', 'empresas', $empresa->id, $datosAnteriores, $empresa->fresh()->toArray());

        return redirect()
            ->route('empresas.show', $empresa)
            ->with('success', 'Empresa aprobada exitosamente.');
    }

    public function reject(Request $request, Empresa $empresa)
    {
        $this->authorize('reject', $empresa);

        $request->validate([
            'comentarios_aprobacion' => 'required|string',
        ]);

        $datosAnteriores = $empresa->toArray();

        $empresa->update([
            'estado' => 'rechazada',
            'aprobado_por' => auth()->user()->id,
            'fecha_aprobacion' => now(),
            'comentarios_aprobacion' => $request->comentarios_aprobacion,
        ]);

        // Notificar a los proveedores asignados
        foreach ($empresa->proveedores as $proveedor) {
            Notificacion::create([
                'user_id' => $proveedor->id,
                'titulo' => 'Empresa rechazada',
                'mensaje' => "La empresa '{$empresa->razon_social}' ha sido rechazada. Motivo: {$request->comentarios_aprobacion}",
                'tipo' => 'warning',
            ]);
        }

        Auditoria::registrar('reject', 'empresas', $empresa->id, $datosAnteriores, $empresa->fresh()->toArray());

        return redirect()
            ->route('empresas.show', $empresa)
            ->with('success', 'Empresa rechazada.');
    }

    public function destroy(Empresa $empresa)
    {
        $this->authorize('delete', $empresa);

        $datosAnteriores = $empresa->toArray();
        $empresa->delete();

        Auditoria::registrar('delete', 'empresas', $empresa->id, $datosAnteriores, null);

        return redirect()
            ->route('empresas.index')
            ->with('success', 'Empresa eliminada exitosamente.');
    }

    public function addTareas(Empresa $empresa)
    {
        $this->authorize('addProjects', $empresa);

        // Validación adicional: solo empresas aprobadas
        if ($empresa->estado != 'aprobada') {
            return redirect()
                ->route('empresas.show', $empresa)
                ->with('error', 'Solo se pueden agregar tareas a empresas aprobadas.');
        }

        // Validación adicional: el proveedor solo puede agregar a sus empresas asignadas
        if (auth()->user()->hasRole('proveedor') && !$empresa->proveedores->contains(auth()->user()->id)) {
            abort(403, 'Solo puedes agregar tareas a tus propias empresas.');
        }

        $operadores = User::role('operador')->where('estado', 'activo')->get();
        $ultimoOrden = $empresa->tareas()->max('orden') ?? 0;

        return view('empresas.add-tareas', compact('empresa', 'operadores', 'ultimoOrden'));
    }

    public function storeTareas(Request $request, Empresa $empresa)
    {
        $this->authorize('addProjects', $empresa);

        if ($empresa->estado != 'aprobada') {
            return redirect()
                ->route('empresas.show', $empresa)
                ->with('error', 'Solo se pueden agregar tareas a empresas aprobadas.');
        }

        $request->validate([
            'tareas' => 'required|array|min:1',
            'tareas.*.titulo' => 'required|string|max:255',
            'tareas.*.descripcion' => 'required|string',
            'tareas.*.fecha_inicio_estimada' => 'required|date',
            'tareas.*.fecha_fin_estimada' => 'required|date|after_or_equal:tareas.*.fecha_inicio_estimada',
            'tareas.*.prioridad' => 'required|in:baja,media,alta,urgente',
            'tareas.*.operador_id' => 'required|exists:users,id',
        ]);

        DB::beginTransaction();

        try {
            $ultimoOrden = $empresa->tareas()->max('orden') ?? 0;
            $tareasCreadas = [];

            foreach ($request->tareas as $index => $tareaData) {
                $tarea = Tarea::create([
                    'empresa_id' => $empresa->id,
                    'titulo' => $tareaData['titulo'],
                    'descripcion' => $tareaData['descripcion'],
                    'fecha_inicio_estimada' => $tareaData['fecha_inicio_estimada'],
                    'fecha_fin_estimada' => $tareaData['fecha_fin_estimada'],
                    'prioridad' => $tareaData['prioridad'],
                    'operador_id' => $tareaData['operador_id'],
                    'orden' => $ultimoOrden + $index + 1,
                    'estado' => 'pendiente',
                ]);

                $tareasCreadas[] = $tarea;
            }

            // Notificar a los operadores asignados
            $operadoresNotificados = [];
            foreach ($tareasCreadas as $tarea) {
                // Enviar correo por cada tarea
                try {
                    $tarea->load('operador', 'empresa');
                    if ($tarea->operador && $tarea->operador->email) {
                        Mail::to($tarea->operador->email)->send(new NuevaTareaAsignada($tarea));
                    }
                } catch (\Exception $e) {
                    // Continue if email fails
                }

                if (!in_array($tarea->operador_id, $operadoresNotificados)) {
                    Notificacion::create([
                        'user_id' => $tarea->operador_id,
                        'titulo' => 'Nuevas tareas asignadas',
                        'mensaje' => "Se te han asignado nuevas tareas en la empresa: {$empresa->razon_social}",
                        'tipo' => 'info',
                    ]);
                    $operadoresNotificados[] = $tarea->operador_id;
                }
            }

            // Notificar a administradores
            $admins = User::role('administrador')->get();
            foreach ($admins as $admin) {
                if ($admin->id != auth()->id()) {
                    Notificacion::create([
                        'user_id' => $admin->id,
                        'titulo' => 'Tareas agregadas a empresa',
                        'mensaje' => 'Se agregaron ' . count($tareasCreadas) . " tareas a la empresa: {$empresa->razon_social}",
                        'tipo' => 'info',
                    ]);
                }
            }

            // Notificar a proveedores
            foreach ($empresa->proveedores as $proveedor) {
                if ($proveedor->id != auth()->id()) {
                    Notificacion::create([
                        'user_id' => $proveedor->id,
                        'titulo' => 'Tareas agregadas a empresa',
                        'mensaje' => 'Se agregaron ' . count($tareasCreadas) . " tareas a su empresa: {$empresa->razon_social}",
                        'tipo' => 'info',
                    ]);
                }
            }

            foreach ($tareasCreadas as $tarea) {
                Auditoria::registrar('create_project_in_series', 'tareas', $tarea->id, null, $tarea->toArray());
            }

            DB::commit();

            if ($request->ajax() || $request->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Se agregaron ' . count($tareasCreadas) . ' tareas exitosamente.'
                ]);
            }

            return redirect()
                ->route('empresas.show', $empresa)
                ->with('success', 'Se agregaron ' . count($tareasCreadas) . ' tareas exitosamente a la empresa.');
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()
                ->back()
                ->with('error', 'Error al agregar tareas: ' . $e->getMessage())
                ->withInput();
        }
    }

    public function modalContent(Empresa $empresa)
    {
        $this->authorize('view', $empresa);
        // Ensure relationships are loaded
        $empresa->load(['proveedor', 'tareas.operador', 'aprobadoPor']);
        return view('empresas.partials.show_modal_content', compact('empresa'));
    }

    public function modalTareasContent(Request $request, Empresa $empresa)
    {
        $this->authorize('view', $empresa);

        // Load all tasks for statistics (if needed by view)
        // If the view relies on $empresa->tareas for stats, keeping this.
        // Optimization: Could use loadCount/aggregates instead, but for now maintaining compatibility.
        $empresa->load('tareas');

        // Query for the list with pagination and sorting
        $query = $empresa->tareas()->getQuery();

        // Filtering
        if ($request->filled('estado') && $request->estado !== 'todos') {
            if ($request->estado === 'no_finalizado') {
                $query->where('estado', '!=', 'finalizado');
            } else {
                $query->where('estado', $request->estado);
            }
        }

        // Sorting
        if ($request->filled('sort_by')) {
            $sortBy = $request->sort_by;
            $order = $request->order === 'desc' ? 'desc' : 'asc';

            // Validate allowed sort columns
            $allowedSorts = ['titulo', 'operador_id', 'prioridad', 'estado', 'fecha_fin_estimada', 'orden'];
            if (in_array($sortBy, $allowedSorts)) {
                if ($sortBy == 'operador_id') {
                    // Sorting by related model (Operador Name) requires join if strictly needed,
                    // or just sort by ID (simple). For simple implementation sort by ID or skip complex join for now.
                    $query->orderBy($sortBy, $order);
                } else {
                    $query->orderBy($sortBy, $order);
                }
            }
        } else {
            $query->orderBy('orden', 'asc');
        }

        $tareas = $query->with('operador')->paginate(5)->withQueryString();

        return view('empresas.partials.tasks_modal_content', compact('empresa', 'tareas'));
    }

    public function modalAddTasksContent(Empresa $empresa)
    {
        $this->authorize('addProjects', $empresa);
        $operadores = User::role('operador')->where('estado', 'activo')->get();
        return view('empresas.partials.add_tasks_modal_content', compact('empresa', 'operadores'));
    }

    public function modalRucs(Empresa $empresa)
    {
        $this->authorize('view', $empresa);
        $empresa->load('otrosRucs');
        return view('empresas.partials.ruc_list', compact('empresa'));
    }

    public function modalRucForm(Empresa $empresa)
    {
        $this->authorize('update', $empresa);
        return view('empresas.partials.ruc_form', compact('empresa'));
    }

    public function modalRucEditForm(EmpresaRuc $empresaRuc)
    {
        $this->authorize('update', $empresaRuc->empresa);
        $empresa = $empresaRuc->empresa;
        return view('empresas.partials.ruc_edit_form', compact('empresaRuc', 'empresa'));
    }

    public function updateRuc(Request $request, EmpresaRuc $empresaRuc)
    {
        $this->authorize('update', $empresaRuc->empresa);

        $request->validate([
            'ruc' => 'required|string|max:11',
            'razon_social' => 'required|string|max:255',
            'serie' => 'nullable|string|max:3',
            'usuario_sol' => 'nullable|string|max:255',
            'clave_sol' => 'nullable|string|max:255',
            'usuario_secundario_dni' => 'nullable|string|max:8',
            'usuario_secundario_user' => 'nullable|string|max:255',
            'usuario_secundario_clave' => 'nullable|string|max:255',
            'clave_certificado' => 'nullable|string|max:255',
        ]);

        $empresaRuc->update($request->all());

        return redirect()->route('empresas.index')->with('success', 'RUC actualizado correctamente.');
    }

    public function destroyRuc(EmpresaRuc $empresaRuc)
    {
        $this->authorize('update', $empresaRuc->empresa);
        $empresaRuc->delete();
        return redirect()->route('empresas.index')->with('success', 'RUC eliminado de la empresa.');
    }
}
