<?php
// app/Http/Controllers/Admin/UserController.php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Auditoria;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Role;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'role:administrador']);
    }

    public function index(Request $request)
    {
        $sort = $request->get('sort', 'created_at');
        $direction = $request->get('direction', 'desc');

        // Validar columnas permitidas
        $allowedSorts = ['username', 'dni', 'nombre', 'email', 'estado', 'created_at'];
        if (!in_array($sort, $allowedSorts)) {
            $sort = 'created_at';
        }

        // Validar dirección
        $direction = in_array(strtolower($direction), ['asc', 'desc']) ? $direction : 'desc';

        $users = User::with('roles')
            ->orderBy($sort, $direction)
            ->paginate(10);

        $roles = Role::all();
        return view('admin.users.index', compact('users', 'roles'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'apellidos' => 'required|string|max:255',
            'dni' => 'required|string|max:8|unique:users',
            'email' => 'required|string|email|max:255|unique:users',
            'username' => 'required|string|max:255|unique:users',
            'password' => 'required|string|min:6|confirmed',
            'role' => 'required|exists:roles,name',
        ]);

        $user = User::create([
            'nombre' => $request->nombre,
            'apellidos' => $request->apellidos,
            'dni' => $request->dni,
            'email' => $request->email,
            'username' => $request->username,
            'password' => Hash::make($request->password),
            'estado' => 'activo',
        ]);

        $user->assignRole($request->role);

        Auditoria::registrar('create', 'users', $user->id, null, $user->toArray());

        return redirect()->route('admin.users.index')
            ->with('success', 'Usuario creado exitosamente.');
    }

    public function update(Request $request, User $user)
    {
        $datosAnteriores = $user->toArray();

        $request->validate([
            'nombre' => 'required|string|max:255',
            'apellidos' => 'required|string|max:255',
            'dni' => 'required|string|max:8|unique:users,dni,' . $user->id,
            'email' => 'required|string|email|max:255|unique:users,email,' . $user->id,
            'username' => 'required|string|max:255|unique:users,username,' . $user->id,
            'password' => 'nullable|string|min:6|confirmed',
            'role' => 'required|exists:roles,name',
        ]);

        $updateData = [
            'nombre' => $request->nombre,
            'apellidos' => $request->apellidos,
            'dni' => $request->dni,
            'email' => $request->email,
            'username' => $request->username,
        ];

        if ($request->filled('password')) {
            $updateData['password'] = Hash::make($request->password);
        }

        $user->update($updateData);
        $user->syncRoles([$request->role]);

        Auditoria::registrar('update', 'users', $user->id, $datosAnteriores, $user->fresh()->toArray());

        return redirect()->route('admin.users.index')
            ->with('success', 'Usuario actualizado exitosamente.');
    }

    public function toggleStatus(User $user)
    {
        if ($user->id == auth()->id()) {
            return back()->with('error', 'No puedes desactivar tu propia cuenta.');
        }

        $datosAnteriores = $user->toArray();
        
        $user->estado = $user->estado === 'activo' ? 'inactivo' : 'activo';
        $user->save();

        Auditoria::registrar('update', 'users', $user->id, $datosAnteriores, $user->fresh()->toArray());

        return back()->with('success', 'Estado del usuario actualizado correctamente.');
    }

    public function destroy(User $user)
    {
        if ($user->id == auth()->user()->id) {
            return redirect()->route('admin.users.index')
                ->with('error', 'No puedes eliminarte a ti mismo.');
        }

        $datosAnteriores = $user->toArray();
        $user->delete();

        Auditoria::registrar('delete', 'users', $user->id, $datosAnteriores, null);

        return redirect()->route('admin.users.index')
            ->with('success', 'Usuario eliminado exitosamente.');
    }
}
